/*
 * StringStack Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>

#define STACK_INC 64
#define STRING_INC 128

struct string
{
   int free, used;
   char *top, *str;
};

struct string *string_make( char * );
void string_free( struct string * );

int string_append( struct string *, char );
int string_prepend( struct string *, char );

int string_concat( struct string *, char * );
int string_precat( struct string *, char * );
int string_assign( struct string *, char * );

int string_merge( struct string *, struct string * );
void string_clear( struct string * );

struct string *string_make( char *o )
{
   struct string *s;

   if (( s = malloc( sizeof( struct string ))) == NULL )
      return NULL;

   if (( s->str = malloc( STRING_INC + 1 )) == NULL )
   {
      free( s );
      return NULL;
   }

   *s->str = '\0';
   s->free = STRING_INC;
   s->used = 0;
   s->top = s->str;

   if ( o != NULL && string_concat( s, o ))
   {
      string_free( s );
      return NULL;
   }

   return s;
}

void string_free( struct string *s )
{
   free( s->str );
   free( s );
}

int string_append( struct string *s, char c )
{
   if ( s->free == 0 )
   {
      if (( s->str = realloc( s->str, s->used + 1 + STRING_INC )) == NULL )
         return 1;

      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
   }

   ++s->used;
   --s->free;
   *s->top++ = c;
   *s->top = '\0';

   return 0;
}

int string_prepend( struct string *s, char c )
{
   char *ptr1, *ptr2;

   if ( s->used == 0 )
   {
      string_append( s, c );
      return 0;
   }

   if ( s->free == 0 )
   {
      if (( s->str = realloc( s->str, s->used + 1 + STRING_INC )) == NULL )
         return 1;

      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
   }

   ptr1 = &s->str[ s->used ];
   ptr2 = &s->str[ s->used + 1 ];

   while( ptr1 >= s->str )
      *ptr2-- = *ptr1--;

   s->str[ 0 ] = c;

   ++s->used;
   ++s->top;
   --s->free;

   return 0;
}

int string_concat( struct string *s, char *c )
{
   while( *c )
      if ( ! s->free )
      {
         if ( string_append( s, *c++ ))
            return 1;
      }
      else
      {
         ++s->used;
         --s->free;
         *s->top++ = *c++;
         *s->top = '\0';
      }

   return 0;
}

int string_assign( struct string *s, char *c )
{
   s->free += s->used;
   s->used = 0;
   s->top = s->str;
   *s->str = '\0';

   if ( string_concat( s, c ))
      return 1;

   return 0;
}

int string_precat( struct string *s, char *c )
{
   int len;
   char *old = NULL, *ptr1, *ptr2;

   for( len = 0, ptr1 = c; *ptr1; ++ptr1 )
      ++len;

   if ( len <= s->free )
   {
      ptr1 = s->top + len;
      ptr2 = s->top;

      while( ptr2 >= s->str )
         *ptr1-- = *ptr2--;

      s->used += len;
      s->free -= len;
      s->top = &s->str[ s->used ];
   }
   else
   {
      old = s->str;

      if (( s->str = malloc( s->used + len + 1 + STRING_INC )) == NULL )
         return 1;

      s->used = s->used + len;
      s->free = STRING_INC;
      s->top = &s->str[ s->used ];
      *s->top = '\0';
   }

   ptr1 = s->str;
   ptr2 = c;

   while( *ptr2 )
      *ptr1++ = *ptr2++;

   if ( old != NULL )
   {
      ptr2 = old;

      while( *ptr2 )
         *ptr1++ = *ptr2++;

      free( old );
   }

   return 0;
}

int string_merge( struct string *a, struct string *b )
{
   if ( string_concat( a, b->str ))
      return 1;

   return 0;
}

void string_clear( struct string *s )
{
   s->free += s->used;
   s->used = 0;
   s->top = s->str;
   *s->top = '\0';
}

unsigned int string_UTF8_len( unsigned char *str )
{
   unsigned int n = 0;
   unsigned char *ptr;

   for( ptr = str; *ptr; ++ptr )
      if (( *ptr & 0xC0 ) != 0x80 )
         ++n;

   return n;
}

