/*
 * Prospero Copyright (c) 2016-2025, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <limits.h>
#include <inttypes.h>
#include <stdio.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <dirent.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>

#include <pwd.h>
#include <grp.h>
#include <time.h>

#include <fcntl.h>
#include <signal.h>
#include <syslog.h>
#include <stdarg.h>

#include <string.h>
#include <ctype.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/dh.h>
#include <openssl/ec.h>
#include <openssl/bio.h>
#include <openssl/x509.h>
#include <openssl/crypto.h>
#include <openssl/tls1.h>

SSL_CTX *ctx;

#define STRING 128

/*
 * Default number of new connections accepted at one time in accept_conn().
 */

#define NEWCONS 1024

/*
 * Number of new events for each descriptor that can be generated processing
 * an event for that descriptor.
 */

#define MAXEVENT 10

/*
 * Size of the default output queue used by kevent().
 */

#define QLEN 50000

/*
 * Size of cache hash tables.
 */

#define HASH_SIZE 128

/*
 * Virtual paths of SCGI and WebSocket requests.
 */

#define SCGI1 "/scgi/"
#define SCGI2 "/scgi2/"
#define SCGI3 "/scgi3/"
#define SCGI4 "/scgi4/"
#define SCGI5 "/scgi5/"

#define WEBSOCKET1 "/websocket/"
#define WEBSOCKET2 "/websocket2/"
#define WEBSOCKET3 "/websocket3/"
#define WEBSOCKET4 "/websocket4/"
#define WEBSOCKET5 "/websocket5/"

/*
 * connection control block.
 */

struct ccb
{
   SSL *ssl;

   int type, error, count, sock, source, fd, filter,
       sock_opposite, source_count, sock_count, source_written;

   off_t size, written;
   unsigned int flags;
   time_t start;

   void ( *state )( struct ccb * ), ( *next )( struct ccb * );

   char *request, *method, *version, *cookie, *expect, *host,
      *origin, *connection, *since, *unmodified, *referer, *length,
      *upgrade, *ws_key, *ws_verno, *content, *location, *disposition,
      *cookie2, *cookie3, *cookie4, *cookie5, *control, *range, *gpc,
      *scgi, *websocket, *path, *agent, *language;

   char *buffer;
   struct ccb **conn;

   struct string *netstring;
} **conns = NULL;

/*
 * Boolean flags.  Values with _FOUND in their names have corresponding
 * stored char or int data extracted from a request header.
 */

#define REQUEST_FOUND           0b1
#define ORIGIN_FOUND            0b10
#define CONNECTION_FOUND        0b100
#define HOST_FOUND              0b1000
#define SINCE_FOUND             0b10000
#define UNMODIFIED_FOUND        0b100000
#define REFERRER_FOUND          0b1000000
#define CLOSE                   0b10000000
#define INACTIVE                0b100000000
#define TIMER                   0b1000000000
#define UPGRADE_FOUND           0b10000000000
#define WS_KEY_FOUND            0b100000000000
#define WS_VER_FOUND            0b1000000000000
#define EXPECT_FOUND            0b10000000000000
#define TYPE_FOUND              0b100000000000000
#define LOCATION_FOUND          0b1000000000000000
#define LENGTH_FOUND            0b10000000000000000
#define COOKIE_FOUND            0b100000000000000000
#define COOKIE2_FOUND           0b1000000000000000000
#define COOKIE3_FOUND           0b10000000000000000000
#define COOKIE4_FOUND           0b100000000000000000000
#define COOKIE5_FOUND           0b1000000000000000000000
#define DISPOSITION_FOUND       0b10000000000000000000000
#define CONTROL_FOUND           0b100000000000000000000000
#define VERNO                   0b1000000000000000000000000
#define RANGE_FOUND             0b10000000000000000000000000
#define GPC_FOUND               0b100000000000000000000000000
#define AGENT_FOUND             0b1000000000000000000000000000
#define KTLS_FOUND              0b10000000000000000000000000000
#define LANG_FOUND              0b100000000000000000000000000000

struct hash
{
   struct hash *next;
   char *key, *value;
} *type_cache[ HASH_SIZE ];

void free_conn( struct ccb * );
int connect_to_server( char * );
char *get_line( int, int );
int set_socket_events( struct ccb *, int );
void downcase( char *, int );
void open_logfile();

void ssl_accept( struct ccb * );
void receive_request( struct ccb * );
void redirect_insecure_request( struct ccb * );
void analyze_request( struct ccb * );
void prepare_static( struct ccb * );

void respond_static( struct ccb * );
void respond_static_ktls( struct ccb * );
void respond_error( struct ccb * );
void respond_header( struct ccb * );

void websocket_connect( struct ccb * );
void websocket_verify( struct ccb * );
void websocket_transfer( struct ccb * );

void scgi_connect( struct ccb * );
void scgi_prepare( struct ccb * );
void scgi_header( struct ccb * );
void scgi_body( struct ccb * );
void scgi_filter( struct ccb  * );

void respond_chunked( struct ccb * );
void respond_unchunked( struct ccb * );

void ( *ACCEPT   )( struct ccb * ) = ssl_accept;
void ( *RECEIVE  )( struct ccb * ) = receive_request;
void ( *INSECURE )( struct ccb * ) = redirect_insecure_request;
void ( *ANALYZE  )( struct ccb * ) = analyze_request;

void ( *RESPOND )( struct ccb * )      = respond_static;
void ( *RESPOND_KTLS )( struct ccb * ) = respond_static_ktls;
void ( *ERROR   )( struct ccb * )      = respond_error;
void ( *HEADER  )( struct ccb * )      = respond_header;
void ( *PREPARE )( struct ccb * )      = prepare_static;

void ( *WS_CONNECT  )( struct ccb * ) = websocket_connect;
void ( *WS_VERIFY   )( struct ccb * ) = websocket_verify;
void ( *WS_TRANSFER )( struct ccb * ) = websocket_transfer;

void ( *SCGI_CONNECT )( struct ccb * ) = scgi_connect;
void ( *SCGI_PREPARE )( struct ccb * ) = scgi_prepare;
void ( *SCGI_HEADER  )( struct ccb * ) = scgi_header;
void ( *SCGI_BODY    )( struct ccb * ) = scgi_body;
void ( *SCGI_FILTER  )( struct ccb * ) = scgi_filter;

void ( *CHUNK   )( struct ccb * ) = respond_chunked;
void ( *NOCHUNK )( struct ccb * ) = respond_unchunked;

/*
 * These must start at 1.  0 means "no error".
 */

#define ERR_BADVERSION        1
#define ERR_BADREQUEST        2
#define ERR_FORBIDDEN         3
#define ERR_BADMETHOD         4
#define ERR_NOTFOUND          5
#define ERR_INTERNAL          6
#define ERR_NOTMODIFIED       7
#define ERR_MODIFIED          8
#define ERR_TOOMANYREQ        9
#define ERR_WSVERSION        10
#define ERR_LENGTHREQ        11
#define ERR_RANGE            12

struct kevent *inqueue, *outqueue;

int IOBUFSIZE = 1024, HALFSIZE = 256, TIMEOUT = 30;

int idx = 0, in = 0, out = 0, qlen = QLEN * MAXEVENT,
   instances = -1, active = 0, total = 0, max_conn = QLEN,
   cache_val = 600, scgi_cross_origin = 0,
   ws_cross_origin = 0, logging = 0;

volatile int sigterm = 0, sighup = 0;

extern char *optarg;
FILE *logfile = NULL;

char *root_resource = "index.html",
     *type_file = "/usr/local/share/prospero/content-types.txt",
     *log_filename = "/var/log/prospero.log",
     *config_file = NULL, *password = NULL, *keyfile = NULL, *chainfile = NULL,

     *server_header =
        "Server: Prospero " VERSION "\r\n"
        "Accept-Ranges: bytes\r\n",

     *language         = NULL,
     *content_language = NULL,
     *language_string  = "Content-Language: %s\r\n",

     *cache_string  = "Cache-Control: max-age=%d, must-revalidate\r\n",
     *cache_control = NULL,

     *hostname     = "",
     *hostname_443 = "",

     *www_hostname     = "",
     *www_hostname_443 = "",

     *websocket1 = NULL, *websocket2 = NULL, *websocket3 = NULL, *websocket4 = NULL, *websocket5 = NULL,
     *scgi1 = NULL, *scgi2 = NULL, *scgi3 = NULL, *scgi4 = NULL, *scgi5 = NULL,
     *ws_guid = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";

int insec_fd = -1, sec_fd = -1, increment = 1024, backlog = NEWCONS * 2;

struct passwd *passwd;
struct group *group;

char *interface = "",
   *grp = "nobody", *user = "nobody", *root = NULL,
   *default_type = "text/plain; charset=utf-8";

char *errstrings[ 16 ] = { "",
     " 505 HTTP Version Not Supported\r\n",
     " 400 Bad Request\r\n",
     " 403 Forbidden\r\n",
     " 405 Method Not Allowed\r\nAllow: GET, HEAD\r\n",
     " 404 Not Found\r\n",
     " 500 Internal Server Error\r\n",
     " 304 Not Modified\r\n",
     " 402 Precondition Failed\r\n",
     " 429 Too Many Requests\r\n"
     " 400 Bad Request\r\nSec-Websocket-Version: 13\r\n",
     " 411 Length Required\r\n",
     " 416 Range Not Satisfiable\r\n"
};

struct string
{
   int free, used;
   char *top, *str;
} *logline = NULL;

#define STRING_TRUNCATE( _s_ ) if (( _s_ )->used )\
  {\
     ( _s_ )->free += ( _s_ )->used;\
     ( _s_ )->used = 0;\
     *( _s_ )->str = '\0';\
     ( _s_ )->top = ( _s_ )->str;\
  }

void *memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
      syslog( LOG_WARNING, "malloc(): %m" );

   return ptr;
}

int string_append( struct string *s, char c )
{
   if ( s->free == 0 )
   {
      s->str = ( char *)realloc( s->str, s->used + 1 + increment );

      if ( s->str == NULL )
      {
         syslog( LOG_WARNING, "realloc(): %m" );
         return 1;
      }

      /* Leave room for end-of-string sentinel */

      s->free = increment;
      s->top = &s->str[ s->used ];
   }

   ++s->used;
   --s->free;
   *s->top++ = c;
   *s->top = '\0';

   return 0;
}

int string_prepend( struct string *s, char c )
{
   char *ptr, *ptr2;

   if ( s->used == 0 )
      return string_append( s, c );

   if ( s->free == 0 )
   {
      s->str = realloc( s->str, s->used + 1 + STRING );

      if ( s->str == NULL )
      {
         syslog( LOG_ERR, "realloc: %s", strerror( errno ));
         return 1;
      }

      /* Leave room for end-of-string sentinel. */

      s->free = STRING - 1;
      s->top = &s->str[ s->used ];
   }

   ptr2 = &s->str[ s->used + 1 ];

   for( ptr = &s->str[ s->used ]; ptr >= s->str; --ptr )
      *ptr2-- = *ptr;

   s->str[ 0 ] = c;

   ++s->used;
   ++s->top;
   --s->free;

   return 0;
}

void log_error()
{
   unsigned long err;
   char buffer[ 128 ];

   while(( err = ERR_get_error()))
   {
      *buffer = '\0';
      ERR_error_string_n( err, buffer, sizeof( buffer ));
      syslog( LOG_ERR, "%s", buffer );
   }
}

void ev_set( int desc, short filter, u_short flags, struct ccb *item )
{
   if ( in >= qlen )
      return;

   inqueue[ in ].ident  = desc;
   inqueue[ in ].filter = filter;
   inqueue[ in ].fflags = 0;
   inqueue[ in ].flags  = flags;
   inqueue[ in ].udata  = item;

   ++in;
}

void set_timer()
{
   if ( in >= qlen )
      return;

   inqueue[ in ].ident  = 1;
   inqueue[ in ].filter = EVFILT_TIMER;
   inqueue[ in ].fflags = NOTE_SECONDS;
   inqueue[ in ].data   = TIMEOUT;
   inqueue[ in ].flags  = EV_ADD;
   inqueue[ in ].udata  = NULL;

   ++in;
}

void delete_timer()
{
   if ( in >= qlen )
      return;

   inqueue[ in ].ident  = 1;
   inqueue[ in ].filter = EVFILT_TIMER;
   inqueue[ in ].fflags = NOTE_SECONDS;
   inqueue[ in ].data   = TIMEOUT;
   inqueue[ in ].flags  = EV_DELETE;
   inqueue[ in ].udata  = NULL;
}

void add_read_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_READ, EV_ADD, item );
}

void add_write_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_WRITE, EV_ADD, item );
}

void stop_read_socket( struct ccb *item )
{
   ev_set( item->sock, EVFILT_READ, EV_DISABLE, item );
}

void start_read_socket( struct ccb *item )
{
   ev_set( item->sock, EVFILT_READ, EV_ENABLE, item );
}

void stop_write_socket( struct ccb *item )
{
   ev_set( item->sock, EVFILT_WRITE, EV_DISABLE, item );
}

void start_write_socket( struct ccb *item )
{
   ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
}

void stop_read_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_READ, EV_DISABLE, item );
}

void start_read_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_READ, EV_ENABLE, item );
}

void stop_write_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_WRITE, EV_DISABLE, item );
}

void start_write_source( struct ccb *item )
{
   ev_set( item->source, EVFILT_WRITE, EV_ENABLE, item );
}

void flip_events( struct ccb *item )
{
   ev_set( item->sock, EVFILT_READ, EV_DISABLE, item );
   ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
}

void unflip_events( struct ccb *item )
{
   ev_set( item->sock, EVFILT_READ, EV_ENABLE, item );
   ev_set( item->sock, EVFILT_WRITE, EV_DISABLE, item );
}

void non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      if ( errno != EBADF )
         exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      if ( errno != EBADF )
         exit( 1 );
   }
}

char *str_dup( char *str, int len )
{
   char *ptr;

   if ( len < 0 )
      for( len = 0, ptr = str; *ptr; ++ptr )
         ++len;

   if (( ptr = ( char *)memory( len + 1 )) != NULL )
   {
      bcopy( str, ptr, len );
      ptr[ len ] = '\0';
   }

   return ptr;
}

struct string *make_string()
{
   struct string *s;

   if (( s = ( struct string *)memory( sizeof( struct string ))) == NULL )
      return s;

   if (( s->str = ( char *)memory( increment + 1 )) == NULL )
   {
      free( s );
      return NULL;
   }

   *s->str = '\0';
   s->free = increment;
   s->used = 0;
   s->top = s->str;

   return s;
}

void string_free( struct string *s )
{
   free( s->str );
   free( s );
}

void sigterm_handler( int signo )
{
   ++sigterm;
}

void sighup_handler( int signo )
{
   ++sighup;
}

void set_options( int argc, char **argv )
{
   int i, len;
   char buffer[ 128 ];

   /*
    * Process the args.
    */

   while(( i = getopt( argc, argv, "wsla:o:b:c:f:m:i:r:u:g:q:")) != -1 )
      switch( i )
      {
         case 'a':
            language = optarg;
            break;

         case 'b':
            IOBUFSIZE = strtol( optarg, NULL, 10 );
            HALFSIZE = IOBUFSIZE / 2;
            break;

         case 'c':
            cache_val = strtol( optarg, NULL, 10 );
            break;

         case 'f':
            config_file = optarg;
            break;

         case 'g':
            grp = optarg;
            break;

         case 'i':
            interface = optarg;
            break;

         case 'l':
            signal( SIGHUP, sighup_handler );
            ++logging;
            break;

         case 'm':
            max_conn = strtol( optarg, NULL, 10 );
            break;

         case 'o':
            TIMEOUT = strtol( optarg, NULL, 10 );
            break;

         case 'r':
            root = optarg;
            break;

         case 's':
            ++scgi_cross_origin;
            break;

         case 'q':
            backlog = strtol( optarg, NULL, 10 );
            break;

         case 'u':
            user = optarg;
            break;

         case 'w':
            ++ws_cross_origin;
            break;
      }

   if ( IOBUFSIZE % 2 )
   {
      syslog( LOG_ERR, "buffer size is not a multiple of 2: %d", IOBUFSIZE );
      exit( 1 );
   }

   if ( IOBUFSIZE < 1024 || IOBUFSIZE > 65534 )
   {
      syslog( LOG_ERR, "buffer size is our of range 1024-65534: %d", IOBUFSIZE );
      exit( 1 );
   }

   if ( TIMEOUT < 10 )
   {
      syslog( LOG_ERR, "idle timeout is < 10 seconds: %d", TIMEOUT );
      exit( 1 );
   }

   if ( cache_val < 0 )
   {
      syslog( LOG_ERR, "cache timeout < 0: %d", cache_val );
      exit( 1 );
   }

   len = snprintf( buffer, sizeof( buffer ), cache_string, cache_val );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated an oversized Cache-Control header. "
                          "The value of -c is too large." );
      exit( 1 );
   }

   if (( cache_control = str_dup( buffer, len )) == NULL )
   {
      syslog( LOG_ERR, "could not allocate Cache-Control header" );
      exit( 1 );
   }

   if ( language != NULL )
   {
      len = snprintf( buffer, sizeof( buffer ), language_string, language );

      if ( len < 0 || len >= sizeof( buffer ))
      {
         if ( len < 0 )
            syslog( LOG_ERR, "snprintf(): %m" );
         else
            syslog( LOG_ERR, "Prospero generated an oversized Content-Language header. "
                             "The value of -a is too long." );
         exit( 1 );
      }

      if (( content_language = str_dup( buffer, len )) == NULL )
      {
         syslog( LOG_ERR, "could not allocate Content-Language header" );
         exit( 1 );
      }
   }

   if ( root == NULL )
   {
      syslog( LOG_ERR, "server root directory not specified" );
      exit( 1 );
   }

   if ( max_conn <= 0 )
   {
      syslog( LOG_ERR, "max connections <= 0: %d", max_conn );
      exit( 1 );
   }

   if (( passwd = getpwnam( user )) == NULL )
   {
      syslog( LOG_ERR, "user \"%s\" does not exist", user );
      exit( 1 );
   }

   if (( group = getgrnam( grp )) == NULL )
   {
      syslog( LOG_ERR, "group \"%s\" does not exist", grp );
      exit( 1 );
   }
}

void change_identity()
{
   if ( setgid( group->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "setgid(): %m" );
      exit( 1 );
   }

   if ( setuid( passwd->pw_uid ) < 0 )
   {
      syslog( LOG_ERR, "setuid(): %m" );
      exit( 1 );
   }
}

void start_listening( int *fd, char *port )
{
   struct addrinfo hints, *res;
   int result;

   bzero( &hints, sizeof( struct addrinfo ));
   hints.ai_flags = AI_PASSIVE;
   hints.ai_socktype = SOCK_STREAM;

   /*
    * If the user has not specified a listening interface, we use
    * the IPv6 wildcard.
    */

   if (( result = getaddrinfo(( *interface ? interface : NULL ), port, &hints, &res )))
   {
      syslog( LOG_ERR, "getaddrinfo(): %s", gai_strerror( result ));
      exit( 1 );
   }

   if ( res == NULL )
   {
      syslog( LOG_ERR, "getaddrinfo(): no interface found" );
      exit( 1 );
   }

   *fd = socket( res->ai_family, res->ai_socktype, res->ai_protocol );

   if ( *fd == -1 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   result = 1;

   if ( setsockopt( *fd, SOL_SOCKET, SO_REUSEPORT, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_REUSEPORT ): %m" );

   if ( setsockopt( *fd, SOL_SOCKET, SO_KEEPALIVE, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_KEEPALIVE ): %m" );

   /*
    * Note we are shutting off this option if we are binding to the IPv6
    * wildcard interface, so we may accept both IPv4 and IPv6 connections.
    */

   result = 0;

   if ( ! *interface && setsockopt( *fd, IPPROTO_IPV6, IPV6_BINDV6ONLY, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( IPV6_BINDV6ONLY ): %m" );

   if ( bind( *fd, res->ai_addr, res->ai_addrlen ) < 0 )
   {
      syslog( LOG_ERR, "bind(): %m" );
      exit( 1 );
   }

   if ( res != NULL )
      freeaddrinfo( res );

   if ( listen( *fd, backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      exit( 1 );
   }

   non_blocking( *fd );
}

char *get_peer_address( int fd, int *port, int *ipv6 )
{
   struct sockaddr_storage sa;
   socklen_t len;
   static char addr[ INET6_ADDRSTRLEN ];
   void *aptr = NULL;

   if ( getpeername( fd, ( struct sockaddr *)&sa, &len ) < 0 )
      syslog( LOG_ERR, "getpeername(): %m" );
   else if ( sa.ss_family == AF_INET6 )
   {
      aptr = &(( struct sockaddr_in6 *)&sa )->sin6_addr;

      if ( ipv6 != NULL )
         *ipv6 = 1;

      if ( port != NULL )
         *port = (( struct sockaddr_in6 *)&sa )->sin6_port;
   }
   else if ( sa.ss_family == AF_INET )
   {
      aptr = &(( struct sockaddr_in *)&sa )->sin_addr;

      if ( ipv6 != NULL )
         *ipv6 = 0;

      if ( port != NULL )
         *port = (( struct sockaddr_in *)&sa )->sin_port;
   }

   if ( aptr == NULL )
      return NULL;

   if ( inet_ntop( sa.ss_family, aptr, addr, INET6_ADDRSTRLEN ) == NULL )
   {
      syslog( LOG_ERR, "inet_ntop(): %m" );
      return NULL;
   }

   return addr;
}

void add_conn( int new, int secure )
{
   struct ccb *ptr;
   static struct ccb **cptr = NULL;
   static int n = 0;

   /*
    * Drop the connection if we have reached max_conn.
    */

   if ( total == max_conn )
   {
      close( new );
      return;
   }

   if (( ptr = memory( sizeof( struct ccb ))) == NULL )
   {
      close( new );
      return;
   }

   bzero( ptr, sizeof( struct ccb ));

   if (( ptr->buffer = memory( IOBUFSIZE )) == NULL )
   {
      close( new );
      free( ptr );
      return;
   }

   bzero( ptr->buffer, IOBUFSIZE );

   if ( cptr == NULL )
      cptr = conns;

AGAIN:
   for( ; n < max_conn; ++n, ++cptr )
      if ( *cptr == NULL )
      {
         *cptr = ptr;
         ptr->conn = cptr;
         break;
      }

   if ( n == max_conn )
   {
      cptr = conns;
      n = 0;
      goto AGAIN;
   }

   ptr->state = secure ? ACCEPT : INSECURE;
   ptr->next = NULL;
   ptr->source = ptr->type = -1;
   ptr->sock = new;

   if ( secure )
   {
      if (( ptr->ssl = SSL_new( ctx )) == NULL )
      {
         log_error();
         close( new );
         free( ptr->buffer );
         free( ptr );
         *cptr = NULL;
         return;
      }

      if ( ! SSL_set_fd( ptr->ssl, ptr->sock ))
      {
         log_error();
         SSL_free( ptr->ssl );
         close( new );
         free( ptr->buffer );
         free( ptr );
         *cptr = NULL;
         return;
      }
   }

   ev_set( ptr->sock, EVFILT_READ, EV_ADD, ptr );
   ev_set( ptr->sock, EVFILT_WRITE, EV_ADD | EV_DISABLE, ptr );
   ptr->start = time( NULL );

   ++active;
   ++total;

   if ( active == 1 )
      set_timer();
}

void ssl_accept( struct ccb *item )
{
   int err;
   BIO *bio;

   if (( err = SSL_accept( item->ssl )) <= 0 )
   {
      err = SSL_get_error( item->ssl, err );

      if ( err == SSL_ERROR_WANT_READ )
      {
         unflip_events( item );
         return;
      }

      if ( err == SSL_ERROR_WANT_WRITE )
      {
         flip_events( item );
         return;
      }

      free_conn( item );
      return;
   }

   if (( bio = SSL_get_wbio( item->ssl )) != NULL && BIO_get_ktls_send( bio ))
      item->flags |= KTLS_FOUND;

   unflip_events( item );
   item->state = RECEIVE;
}

void clear_events( struct ccb *item )
{
   int i;

   for( i = idx + 1; i < out; ++i )
      if ( outqueue[ i ].udata == item )
         outqueue[ i ].ident = 0;
}

void clear_conn( struct ccb *item )
{
   char ***hptr, **headers[] = {
      &item->host,
      &item->connection,
      &item->referer,
      &item->origin,
      &item->version,
      &item->method,
      &item->request,
      &item->ws_verno,
      &item->ws_key,
      &item->upgrade,
      &item->cookie,
      &item->expect,
      &item->length,
      &item->content,
      &item->location,
      &item->cookie2,
      &item->cookie3,
      &item->cookie4,
      &item->cookie5,
      &item->control,
      &item->disposition,
      &item->range,
      &item->path,
      &item->since,
      &item->unmodified,
      &item->gpc,
      &item->agent,
      &item->language,
      NULL
   };

   clear_events( item );

   if ( item->source > 0 )
   {
      close( item->source );
      item->source = -1;
   }

   for( hptr = headers; *hptr != NULL; ++hptr )
      if ( **hptr != NULL )
      {
         free( **hptr );
         **hptr = NULL;
      }

   if ( item->netstring != NULL )
   {
      string_free( item->netstring );
      item->netstring = NULL;
   }

   if ( ! item->sock || item->ssl == NULL )
      return;

   item->flags = 0;
   *item->buffer = '\0';

   item->count = item->error = 0;

   item->state = RECEIVE;
   item->next  = NULL;

   item->size = item->written = 0;
   item->type = -1;

   item->start = time( NULL );
   unflip_events( item );

   if ( SSL_has_pending( item->ssl ))
      receive_request( item );
}

void free_conn( struct ccb *item )
{
   --total;
   *item->conn = NULL;

   if ( item->ssl != NULL )
   {
      SSL_shutdown( item->ssl );
      SSL_free( item->ssl );
   }

   close( item->sock );
   item->sock = 0;

   if ( item->state == NULL || item->state != WS_TRANSFER )
   {
      --active;

      if ( ! active )
         delete_timer();
   }

   clear_conn( item );
   free( item->buffer );
   free( item );
}

int password_callback( char *buf, int size, int rw, void *unused )
{
   char *ptr1, *ptr2;
   int len;

   for( len = 0, ptr1 = password; *ptr1; ++ptr1 )
      ++len;

   if ( len + 1 > size )
   {
      *buf = '\0';
      syslog( LOG_ERR, "TLS password is too long for OpenSSL" );
      return 0;
   }

   ptr1 = password;
   ptr2 = buf;

   while( *ptr1 )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';

   return len;
}

void init_tls()
{
   long options = SSL_OP_NO_TLSv1_2 |
                  SSL_OP_NO_SSLv2 |
                  SSL_OP_NO_SSLv3 |
                  SSL_OP_NO_TLSv1 |
                  SSL_OP_NO_TLSv1_1 |
                  SSL_OP_NO_COMPRESSION |
                  SSL_OP_SINGLE_DH_USE |
                  SSL_OP_CIPHER_SERVER_PREFERENCE |
                  SSL_OP_IGNORE_UNEXPECTED_EOF |
                  SSL_OP_ENABLE_KTLS;

   static char context[ SSL_MAX_SSL_SESSION_ID_LENGTH ];
   int len;

   SSL_load_error_strings();
   SSL_library_init();

   if (( ctx = SSL_CTX_new( TLS_server_method() )) == NULL )
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_ecdh_auto( ctx, 1 ))
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_options( ctx, options ))
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_clear_options( ctx, SSL_OP_LEGACY_SERVER_CONNECT |
                                      SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION ))
   {
      log_error();
      exit( 1 );
   }

   SSL_CTX_set_default_passwd_cb( ctx, password_callback );

   if ( ! SSL_CTX_use_PrivateKey_file( ctx, keyfile, SSL_FILETYPE_PEM ))
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_use_certificate_chain_file( ctx, chainfile ))
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_check_private_key( ctx ))
   {
      log_error();
      exit( 1 );
   }

   len = snprintf( context, sizeof( context ), "prospero-%u", getpid() );

   if ( len < 0 || len >= sizeof( context ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized TLS session id context. "
                          "This shouldn't happen!" );
      exit( 1 );
   }

   if ( ! SSL_CTX_set_session_id_context( ctx, ( const unsigned char *)context, len ))
   {
      log_error();
      exit( 1 );
   }

   if ( ! SSL_CTX_set_session_cache_mode( ctx, SSL_SESS_CACHE_SERVER ))
   {
      log_error();
      exit( 1 );
   }

   SSL_CTX_set_verify( ctx, SSL_VERIFY_NONE, NULL );
}

char **extract_request( struct ccb *item )
{
   static char *results[ 3 ];
   char *ptr, *ptr2;
   int count;

   results[ 0 ] = results[ 1 ] = results[ 2 ] = NULL;

   /*
    * Request method.
    */

   for( count = 0, ptr2 = ptr = item->buffer; *ptr && *ptr != ' ' && *ptr != '\t'; ++ptr )
      ++count;

   if (( results[ 0 ] = str_dup( ptr2, count )) == NULL )
      return NULL;

   while( *ptr && ( *ptr == ' ' || *ptr == '\t' ))
      ++ptr;

   /*
    * Resource name.
    */

   for( count = 0, ptr2 = ptr; *ptr && *ptr != ' ' && *ptr != '\t'; ++ptr )
      ++count;

   if (( results[ 1 ] = str_dup( ptr2, count )) == NULL )
   {
      free( results[ 0 ] );
      return NULL;
   }

   while( *ptr && ( *ptr == ' ' || *ptr == '\t' ))
      ++ptr;

   /*
    * Protocol.
    */

   for( count = 0, ptr2 = ptr; *ptr && *ptr != ' ' && *ptr != '\t'; ++ptr )
      ++count;

   if (( results[ 2 ] = str_dup( ptr2, count )) == NULL )
   {
      free( results[ 0 ] );
      free( results[ 1 ] );
      return NULL;
   }

   if ( ! *results[ 0 ] || ! *results[ 1 ] || ! *results [ 2 ] )
   {
      free( results[ 0 ] );
      free( results[ 1 ] );
      free( results[ 2 ] );

      item->state = ERROR;
      item->error = ERR_BADREQUEST;
      item->flags |= CLOSE;

      return NULL;
   }

   return results;
}

char *extract_header( char *pattern, char *string, int *err )
{
   char *ptr1, *ptr2;
   int count;

   ptr1 = pattern;
   ptr2 = string;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( *ptr1 )
      return NULL;

   while( *ptr2 && ( *ptr2 == ' ' || *ptr2 == '\t' ))
      ++ptr2;

   if ( ! *ptr2 )
      return NULL;

   for( count = 0, ptr1 = ptr2; *ptr1; ++ptr1 )
      ++count;

   if (( ptr1 = str_dup( ptr2, count )) == NULL )
   {
      ++*err;
      return NULL;
   }

   return ptr1;
}

int grab_request( struct ccb *item )
{
   char **matches;

   if (( matches = extract_request( item )) == NULL )
      return 1;

   item->flags  |= REQUEST_FOUND;
   item->method  = matches[ 0 ];
   item->request = matches[ 1 ];
   item->version = matches[ 2 ];

   return 0;
}

int extract_headers( struct ccb *item )
{
   char *match, **sptr, ***hptr;
   unsigned int *iptr;
   int alloc_err = 0;

   static char *strings[] = {
      "host:",
      "connection:",
      "if-modified-since:",
      "if-unmodified-since:",
      "referer:",
      "origin:",
      "upgrade:",
      "sec-websocket-key:",
      "sec-websocket-version:",
      "cookie:",
      "expect:",
      "content-type:",
      "content-length:",
      "range:",
      "sec-gpc:",
      "user-agent:",
      "accept-language:",
      NULL
   };

   unsigned int flags[] = {
      HOST_FOUND,
      CONNECTION_FOUND,
      SINCE_FOUND,
      UNMODIFIED_FOUND,
      REFERRER_FOUND,
      ORIGIN_FOUND,
      UPGRADE_FOUND,
      WS_KEY_FOUND,
      WS_VER_FOUND,
      COOKIE_FOUND,
      EXPECT_FOUND,
      TYPE_FOUND,
      LENGTH_FOUND,
      RANGE_FOUND,
      GPC_FOUND,
      AGENT_FOUND,
      LANG_FOUND
   };

   char **headers[] = {
      &item->host,
      &item->connection,
      &item->since,
      &item->unmodified,
      &item->referer,
      &item->origin,
      &item->upgrade,
      &item->ws_key,
      &item->ws_verno,
      &item->cookie,
      &item->expect,
      &item->content,
      &item->length,
      &item->range,
      &item->gpc,
      &item->agent,
      &item->language
   };

   downcase( item->buffer, 1 );

   for( sptr = strings, iptr = flags, hptr = headers;
        *sptr != NULL;
        ++sptr, ++iptr, ++hptr )
   {
      if (( item->flags & *iptr ) || *item->buffer != **sptr )
         continue;

      if (( match = extract_header( *sptr, item->buffer, &alloc_err )) == NULL )
      {
         if ( alloc_err )
            return 1;

         continue;
      }

      item->flags |= *iptr;
      **hptr = match;
      break;
   }

   return 0;
}

int grab_line( struct ccb *item )
{
   int r;
   char c;

   if ( ! ( item->flags & INACTIVE ))
   {
      item->count = 0;
      *item->buffer = '\0';
   }

   for( ; ; )
   {
      r = SSL_read( item->ssl, &c, 1 );

      if ( r <= 0 )
      {
         int err = set_socket_events( item, r );

         if ( err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE )
         {
            item->flags |= INACTIVE;
            return 2;
         }

         return 1;
      }

      /*
       * We don't want the line terminators.
       */

      if ( c == 13 )
         continue;

      if ( c == 10 )
         break;

      /*
       * New character overwrites previous string terminator.  New string
       * terminator added.
       */

      item->buffer[ item->count++ ] = c;

      if ( item->count >= IOBUFSIZE )
         return 3;

      item->buffer[ item->count ] = '\0';
   }

   item->flags &= ~INACTIVE;
   return 0;
}

void downcase( char *line, int keyword )
{
   char *ptr;

   for( ptr = line; *ptr; ++ptr )
      if ( keyword && *ptr == ':' )
         break;
      else if ( *ptr >= 65 && *ptr <= 90 )
         *ptr += 32;
}

int process_line( struct ccb *item )
{
   if ( ! ( item->flags & REQUEST_FOUND ))
   {
      if ( grab_request( item ))
      {
         if ( item->error )
            flip_events( item );
         else
            free_conn( item );

         return 1;
      }

      return 0;
   }

   if ( extract_headers( item ))
   {
      free_conn( item );
      return 1;
   }

   return 0;
}

void receive_request( struct ccb *item )
{
   int notready;

   for( ; ; )
   {
      notready = grab_line( item );

      /*
       * 0 is success.
       * 1 is premature EOF or error.  Connection has been dropped in set_socket_events().
       * 2 is SSL_ERROR_WANT_READ or SSL_ERR0R_WANT_WRITE on non-blocking socket.
       * 3 is a overlong request header.  We drop connection.
       */

      switch( notready )
      {
         case 0:
            break;

         case 1:
         case 2:
            return;

         case 3:
            free_conn( item );
            return;
      }

      if ( *item->buffer != '\0' )
      {
         if ( process_line( item ))
            return;

         continue;
      }

      if ( item->method == NULL )
      {
         free_conn( item );
         return;
      }

      item->count = 0;
      flip_events( item );
      analyze_request( item );
      return;
   }
}

char *date_header()
{
   static char buffer[ 64 ];
   static time_t t = 0, ot = 0;
   struct tm *lt;

   if (( t = time( NULL )) != ot )
   {
      ot = t;
      lt = gmtime( &t );
      strftime( buffer, sizeof( buffer ), "Date: %a, %d %b %Y %H:%M:%S %Z\r\n", lt );
   }

   return buffer;
}

int add_request( struct ccb *item, struct string *s )
{
   char *ptr;

   if ( string_append( s, '.' ) || string_append( s, '/' ))
      return 1;

   for( ptr = item->request; *ptr; ++ptr )
      if ( string_append( s, *ptr ))
         return 1;

   return 0;
}

int check_permission( struct ccb *item, struct stat *st )
{
   if ( ! S_ISREG( st->st_mode ))
   {
      item->state = ERROR;
      item->error = ERR_NOTFOUND;
      return 1;
   }

   /*
    * Don't send any file which is executable.
    */

   if (( S_IXUSR & st->st_mode ) || ( S_IXGRP & st->st_mode ) || S_IXOTH & st->st_mode )
   {
      item->state = ERROR;
      item->error = ERR_NOTFOUND;
      return 1;
   }

   /*
    * Don't send files not owned by our user or group.  The web server is
    * not permitted to access anyone else's files regardless of
    * permissions.
    */

   if ( st->st_uid != passwd->pw_uid && st->st_gid != group->gr_gid )
   {
      item->state = ERROR;
      item->error = ERR_NOTFOUND;
      return 1;
   }

   /*
    * Don't send files that do not have read perms for our user or group.
    */

   if ( ! ( S_IRUSR & st->st_mode ) && ! ( S_IRGRP & st->st_mode ))
   {
      item->state = ERROR;
      item->error = ERR_NOTFOUND;
      return 1;
   }

   return 0;
}

int set_modified( struct ccb *item, char *modified, int mlen, struct stat *st )
{
   int len;
   struct tm *lt;

   lt = gmtime( &st->st_ctime );
   len = strftime( modified, mlen, "Last-Modified: %a, %d %b %Y %H:%M:%S %Z\r\n", lt );

   if ( item->since != NULL )
   {
      char *ptr1, *ptr2;

      len -= 17;
      ptr1 = item->since;
      ptr2 = &modified[ 15 ];

      while( len && *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         --len;
         ++ptr1;
         ++ptr2;
      }

      if ( ! len )
      {
         item->state = ERROR;
         item->error = ERR_NOTMODIFIED;
         return 1;
      }
   }
   else if ( item->unmodified != NULL )
   {
      char *ptr1, *ptr2;

      len -= 17;
      ptr1 = item->unmodified;
      ptr2 = &modified[ 15 ];

      while( len && *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         --len;
         ++ptr1;
         ++ptr2;
      }

      if ( len )
      {
         item->state = ERROR;
         item->error = ERR_MODIFIED;
         return 1;
      }
   }

   return 0;
}

void insert_type( char *name, char *value )
{
   unsigned int key;
   char *rptr;
   struct hash *ptr = NULL, *ptr2 = NULL, **hptr;

   for( key = 2166136261, rptr = name; *rptr; ++rptr )
      key = ( key * 16777619 ) ^ *rptr;

   key %= HASH_SIZE;
   ptr2 = type_cache[ key ];

   for( ptr = ptr2; ptr != NULL; ptr = ptr->next )
   {
      char *ptr3, *ptr4;

      ptr3 = ptr->key;
      ptr4 = name;

      while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
      {
         ++ptr3;
         ++ptr4;
      }

      if ( ! *ptr3 && ! *ptr4 )
         break;

      ptr2 = ptr;
   }

   /*
    * Duplicate keys are ignored.
    */

   if ( ptr != NULL )
   {
      free( name );
      free( value );
      return;
   }

   hptr = ( ptr2 == NULL ? &type_cache[ key ] : &ptr2->next );

   if (( *hptr = memory( sizeof( struct hash ))) == NULL )
      exit( 1 );

   ( *hptr )->next = NULL;
   ( *hptr )->key = name;
   ( *hptr )->value = value;
}

char *lookup_type( char *item )
{
   unsigned int key;
   char *rptr;
   struct hash *ptr;

   for( rptr = item; *rptr; ++rptr )
      ;

   while( --rptr >= item )
      if ( *rptr == '.' )
      {
         ++rptr;
         break;
      }

   if ( rptr < item )
      return default_type;

   item = rptr;

   for( key = 2166136261; *rptr; ++rptr )
      key = ( key * 16777619 ) ^ *rptr;

   key %= HASH_SIZE;

   if ( type_cache[ key ] == NULL )
      return default_type;

   ptr = type_cache[ key ];

   do
   {
      char *ptr3, *ptr4;

      ptr3 = ptr->key;
      ptr4 = item;

      while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
      {
         ++ptr3;
         ++ptr4;
      }

      if ( ! *ptr3 && ! *ptr4 )
         break;

      ptr = ptr->next;
   }
   while( ptr != NULL );

   if ( ptr == NULL )
      return default_type;

   return ptr->value;
}

void init_content_types()
{
   char *ptr = NULL, *suffix = NULL, *type = NULL;
   int fd;

   if (( fd = open( type_file, O_RDONLY )) < 0 )
   {
      syslog( LOG_ERR, "could not open type file %s", type_file );
      exit( 1 );
   }

   bzero( type_cache, sizeof( struct hash * ) * HASH_SIZE );

   while(( ptr = get_line( fd, 0 )) != NULL )
   {
      if (( suffix = str_dup( ptr, -1 )) == NULL )
      {
         syslog( LOG_ERR, "ran out of memory reading type file %s", type_file );
         exit( 1 );
      }

      if (( ptr = get_line( fd, 0 )) == NULL )
      {
         syslog( LOG_ERR, "type file %s contains a suffix %s with no type", type_file, suffix );
         exit( 1 );
      }

      if (( type = str_dup( ptr, -1 )) == NULL )
      {
         syslog( LOG_ERR, "ran out of memory reading type file %s", type_file );
         exit( 1 );
      }

      insert_type( suffix, type );
   }

   if ( ptr == NULL && type == NULL )
   {
      syslog( LOG_ERR, "type file %s is empty", type_file );
      exit( 1 );
   }

   get_line( -1, 0 );
   close( fd );
}

int open_file( struct ccb *item, char *filename )
{
   int fd;

   if (( fd = open( filename, O_RDONLY )) < 0 )
   {
      item->state = ERROR;

      if ( errno == EACCES || errno == ENOENT || errno == ENOTDIR )
         item->error = ERR_NOTFOUND;
      else
      {
         syslog( LOG_ERR, "open(): %m" );
         item->error = ERR_INTERNAL;
      }

      return -1;
   }

   return fd;
}

int check_for_allowable( struct ccb *item )
{
   char **ptr, *ptr1, *ptr2, *ptr3, *suffixes[] = { "html", "webmanifest", "txt", "ico", "pdf", "epub", "bz", "bz2", "tgz", "txz", "gz", "gzip", "zip", NULL };
   char *names[] = { "/android-chrome-192x192.png", "/android-chrome-512x512.png", "/apple-touch-icon.png", "/favicon-16x16.png", "/favicon-32x32.png", NULL };

   for( ptr = names; *ptr != NULL; ++ptr )
   {
      ptr1 = item->request;
      ptr2 = *ptr;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 && ! *ptr2 )
         return 1;
   }

   for( ptr2 = item->request; *ptr2; ++ptr2 )
      ;

   for( ; ptr2 >= item->request && *ptr2 != '.'; --ptr2 )
      ;

   if ( ptr2 >= item->request )
   {
      ++ptr2;

      for( ptr = suffixes; *ptr != NULL; ++ptr )
      {
         ptr1 = *ptr;
         ptr3 = ptr2;

         while( *ptr1 && *ptr3 && *ptr1 == *ptr3 )
         {
            ++ptr1;
            ++ptr3;
         }

         if ( ! *ptr1 && ! *ptr3 )
            return 1;
      }
   }

   return 0;
}

char *get_local_address( struct ccb *item )
{
   static char addr[ INET6_ADDRSTRLEN ];
   struct sockaddr_storage sa;
   socklen_t len;
   void *aptr = NULL;

   len = sizeof( struct sockaddr_storage );

   if ( getsockname( item->sock, ( struct sockaddr *)&sa, &len ) < 0 )
      syslog( LOG_ERR, "getsockname(): %m" );
   else if ( sa.ss_family == AF_INET6 )
      aptr = &(( struct sockaddr_in6 *)&sa )->sin6_addr;
   else if ( sa.ss_family == AF_INET )
      aptr = &(( struct sockaddr_in *)&sa )->sin_addr;

   if ( aptr == NULL )
      return NULL;

   if ( inet_ntop( sa.ss_family, aptr, addr, INET6_ADDRSTRLEN ) == NULL )
   {
      syslog( LOG_ERR, "inet_ntop(): %m" );
      return NULL;
   }

   return addr;
}

int check_address( struct ccb *item, char *header )
{
   char *addr, *ptr1, *ptr2, *ptr3, *ptr4;
   int tries, ipv6 = 0;

   if (( addr = get_local_address( item )) == NULL )
      return 1;

   ptr1 = "https://";
   ptr2 = header;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( *ptr1 )
      ptr2 = header;

   if ( *ptr2 == '[' )
   {
      ++ipv6;
      ++ptr2;
   }

   tries = 0;
   ptr1 = addr;
   ptr4 = ptr2;

AGAIN:
   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 )
   {
      if ( ! *ptr2 )
         return 0;

      if ( ! ipv6 )
      {
         if ( *ptr2 == ':' || *ptr2 == '/' )
            return 0;
      }
      else if ( *ptr2 == ']' )
         return 0;
   }

   if ( ++tries == 2 )
      return 1;

   ptr1 = addr;
   ptr3 = "::ffff:";

   while( *ptr1 && *ptr3 && *ptr1 == *ptr3 )
   {
      ++ptr1;
      ++ptr3;
   }

   if ( ! *ptr3 )
   {
      ptr2 = ptr4;
      ipv6 = 0;
      goto AGAIN;
   }

   return 1;
}

int reject_other_hostnames( struct ccb *item )
{
   char *ptr1, *ptr2, **ptr, *names[] = { hostname, www_hostname, hostname_443, www_hostname_443, NULL };

   if ( item->host == NULL )
      return 0;

   for( ptr = names; *ptr != NULL; ++ptr )
   {
      ptr1 = *ptr;
      ptr2 = "https://";

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      ptr2 = item->host;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 && ( ! *ptr2 || *ptr2 == ':' ))
         return 0;
   }

   if ( ! check_address( item, item->host ))
      return 0;

   item->state = ERROR;
   item->error = ERR_BADREQUEST;
   return 1;
}

int reject_cross_origin( struct ccb *item, int file )
{
   char *ptr1, *ptr2, **ptr, *header, *names[] = { hostname, www_hostname, hostname_443, www_hostname_443, NULL };

   if ( file && check_for_allowable( item ))
      return 0;

   if ( item->origin == NULL && item->referer == NULL )
   {
      item->state = ERROR;
      item->error = ERR_BADREQUEST;
      return 1;
   }

   header = ( item->origin == NULL ? item->referer : item->origin );

   for( ptr = names; *ptr != NULL; ++ptr )
   {
      ptr1 = *ptr;
      ptr2 = header;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 )
      {
         if ( ! *ptr2 || *ptr2 == '/' )
            return 0;

         if (( *ptr == hostname || *ptr == www_hostname ) && *ptr2 == ':' )
            return 0;
      }
   }

   if ( ! check_address( item, header ))
      return 0;

   item->state = ERROR;
   item->error = ERR_FORBIDDEN;
   return 1;
}

int verify_range( struct ccb *item, struct stat *st_ptr, off_t *idx1, off_t *idx2 )
{
   char *ptr1, *ptr2, *bytes = "bytes=", lower[ 16 ], upper[ 16 ];
   int len;

   ptr1 = item->range;
   ptr2 = bytes;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( *ptr2 )
   {
      item->state = ERROR;
      item->error = ERR_RANGE;
      return 0;
   }

   len = 0;
   ptr2 = ptr1;
   ptr1 = lower;

   while( len < sizeof( lower ) - 1 && *ptr2 >= 48 && *ptr2 <= 57 )
   {
      *ptr1++ = *ptr2++;
      ++len;
   }

   *ptr1 = '\0';

   if ( *ptr2 != '-' )
   {
      item->state = ERROR;
      item->error = ERR_RANGE;
      return 0;
   }

   ptr1 = upper;
   ++ptr2;
   len = 0;

   while( len < sizeof( upper ) - 1 && *ptr2 >= 48 && *ptr2 <= 57 )
   {
      *ptr1++ = *ptr2++;
      ++len;
   }

   *ptr1 = '\0';

   if ( *ptr2 || ( ! *lower && ! *upper ))
   {
      item->state = ERROR;
      item->error = ERR_RANGE;
      return 0;
   }

   *idx1 = strtoll( lower, NULL, 10 );
   *idx2 = strtoll( upper, NULL, 10 );

   if ( ! *idx1 && ! *lower )
   {
      *idx1 = st_ptr->st_size - *idx2;
      *idx2 = st_ptr->st_size - 1;
   }
   else if ( ! *idx2 && ! *upper )
      *idx2 = st_ptr->st_size - 1;

   if ( *idx1 >= st_ptr->st_size || *idx2 >= st_ptr->st_size || *idx2 < *idx1 )
   {
      item->state = ERROR;
      item->error = ERR_RANGE;
      return 0;
   }

   return *idx2 - *idx1 + 1;
}

void prepare_open( struct ccb *item )
{
   struct string *s = NULL;

   if ( reject_cross_origin( item, 1 ))
      return;

   if (( s = make_string()) == NULL )
   {
      free_conn( item );
      return;
   }

   if ( add_request( item, s ))
   {
      string_free( s );
      free_conn( item );
      return;
   }

   item->path = s->str;
   free( s );

   if (( item->source = open_file( item, item->path )) < 0 )
   {
      free_conn( item );
      return;
   }

   /*
    * We continue to let socket write events drive the transaction because
    * if s->str names an empty file, item->source will never generate a
    * read event.
    */

   item->state = PREPARE;
}

void prepare_static( struct ccb *item )
{
   char *ptr1, *ptr2, type[ 256 ], size[ 256 ], modified[ 256 ], range[ 256 ];
   struct stat st;
   off_t rlen = 0, idx1 = 0, idx2 = 0;
   int len;

   if ( fstat( item->source, &st ) < 0 )
   {
      syslog( LOG_ERR, "fstat(): %m" );
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( check_permission( item, &st ))
      return;

   if (( item->flags & RANGE_FOUND ) && ! ( rlen = verify_range( item, &st, &idx1, &idx2 )))
      return;

   if ( set_modified( item, modified, sizeof( modified ), &st ))
      return;

   if ( idx1 && lseek( item->source, idx1, SEEK_SET ) < 0 )
   {
      item->state = ERROR;
      item->error = ERR_RANGE;
      return;
   }

   ptr1 = lookup_type( item->path );
   len = snprintf( type, sizeof( type ), "Content-Type: %s\r\n", ptr1 );

   if ( len < 0 || len >= sizeof( type ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Item in content-types.txt is too long. "
                          "Type specification Lines cannot be longer than 239 bytes." );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   len = snprintf( size, sizeof( size ), "Content-Length: %lu\r\n",
                   ( item->type ? ( rlen ? ( long unsigned int )rlen :
                                     ( long unsigned int )st.st_size ) :
                                     ( long unsigned int )0 ));

   if ( len < 0 || len >= sizeof( size ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated an oversized Content-Length header line. "
                          "This should happen unless the maximum file size on the system "
                          "is beyond humungous." );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( ! rlen )
      *range = '\0';
   else
   {
      len = snprintf( range, sizeof( range ), "Content-Range: bytes %lu-%lu/%lu\r\n",
                      ( long unsigned int )idx1, ( long unsigned int )idx2,
                      ( long unsigned int )st.st_size );

      if ( len < 0 || len >= sizeof( range ))
      {
         if ( len < 0 )
            syslog( LOG_ERR, "snprintf(): %m" );
         else
            syslog( LOG_ERR, "Prospero generated an oversized Content-Range header line. "
                             "This should happen unless the maximum file size on the system "
                             "is beyond humungous." );

         item->state = ERROR;
         item->error = ERR_INTERNAL;
         return;
      }

   }

   ptr2 = "font/";
   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   item->count = snprintf( item->buffer, IOBUFSIZE, "%s %s\r\n%s%s%s%s%s%s%s%s\r\n",
                           item->version, ( rlen ? "206 Partial Content" : "200 OK" ),
                           range, type, size,
                           modified, ( ! *ptr2 ? "" : cache_control ),
                           ( content_language == NULL ? "" : content_language ),
                           server_header,
                           date_header() );

   if ( item->count < 0 || item->count >= IOBUFSIZE )
   {
      if ( item->count < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized response header. "
                          "Dropping connection.  See -b option in manual." );
      free_conn( item );
      return;
   }

   item->written = idx1;
   item->size    = ( rlen ? rlen : st.st_size );
   item->state   = HEADER;
   item->next    = ( item->type ? ( item->flags & KTLS_FOUND ? RESPOND_KTLS : RESPOND ) : NULL );
   item->start   = time( NULL );
}

int set_socket_events( struct ccb *item, int written )
{
   int err = SSL_get_error( item->ssl, written );

   switch( err )
   {
      case SSL_ERROR_WANT_READ:
         unflip_events( item );
         break;

      case SSL_ERROR_WANT_WRITE:
         flip_events( item );
         break;

      case SSL_ERROR_SYSCALL:
      case SSL_ERROR_SSL:
         SSL_free( item->ssl );
         item->ssl = NULL;

      /* Fall Through */

      default:
         log_error();
         free_conn( item );
   }

   return err;
}

void respond_unchunked( struct ccb *item )
{
   int result;
   off_t written;

   item->start = time( NULL );

   if ( ! item->count )
   {
      switch(( result = read( item->source, item->buffer, IOBUFSIZE )))
      {
         case 0:
            (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
            return;

         case -1:
            if ( errno == EWOULDBLOCK )
               goto AGAIN;

            free_conn( item );
            return;

         default:
            item->count = result;
            stop_read_source( item );
      }
   }

   if (( written = SSL_write( item->ssl, item->buffer, item->count )) <= 0 )
   {
      set_socket_events( item, written );
      return;
   }

   item->count = 0;

AGAIN:
   stop_read_socket( item );
   stop_write_socket( item );
   start_read_source( item );
}

void respond_chunked( struct ccb *item )
{
   int result, len;
   off_t written;

   item->start = time( NULL );

   if ( ! item->count )
   {
      if (( result = read( item->source, &item->buffer[ 6 ], IOBUFSIZE - 8 )) < 0 )
      {
         if ( errno == EWOULDBLOCK )
            goto AGAIN;

         free_conn( item );
         return;
      }

      stop_read_source( item );

      if ( ! result )
         item->size = 0;

      item->buffer[ result + 6 ] = '\r';
      item->buffer[ result + 7 ] = '\n';

      item->count = result + 8;

      /* Hard-coded 4 hex digits limits IOBUFSIZE to 65534 */

      len = snprintf( item->buffer, 5, "%.4X", result );

      if ( len < 0 || len >= 5 )
      {
         syslog( LOG_ERR, ( len < 0 ? "snprintf(): %m" : "IOBUFSIZE > 64534" ));
         free_conn( item );
         return;
      }

      item->buffer[ 4 ] = '\r';
      item->buffer[ 5 ] = '\n';
   }

   if (( written = SSL_write( item->ssl, item->buffer, item->count )) <= 0 )
   {
      set_socket_events( item, written );
      return;
   }

   if ( ! item->size )
   {
      (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
      return;
   }

   item->count = 0;

AGAIN:
   stop_read_socket( item );
   stop_write_socket( item );
   start_read_source( item );
}

int grab_scgi_line( struct ccb *item )
{
   int r;
   char c;

   if ( ! ( item->flags & INACTIVE ))
   {
      item->count = 0;
      *item->buffer = '\0';
   }

   for( ; ; )
   {
      r = read( item->source, &c, 1 );

      if ( r <= 0 )
      {
         if ( r < 0 && errno == EWOULDBLOCK )
         {
            item->flags |= INACTIVE;
            return 2;
         }

         return 1;
      }

      /*
       * We don't want the line terminators.
       */

      if ( c == 13 )
         continue;

      if ( c == 10 )
         break;

      /*
       * New character overwrites previous string terminator.  New string
       * terminator added.
       */

      item->buffer[ item->count++ ] = c;

      if ( item->count >= IOBUFSIZE )
      {
         syslog( LOG_ERR, "An SCGI response header line is too long. "
                          "Lines must be no longer than the value of the -b option." );
         return 3;
      }

      item->buffer[ item->count ] = '\0';
   }

   item->flags &= ~INACTIVE;
   return 0;
}

int extract_scgi_headers( struct ccb *item )
{
   char *match, **sptr, ***hptr;
   unsigned int *iptr;
   int alloc_err = 0;

   static char *strings[] = {
      "content-type:",
      "content-length:",
      "content-disposition:",
      "content-language:",
      "cache-control:",
      "location:",
      "set-cookie:",
      "set-cookie:",
      "set-cookie:",
      "set-cookie:",
      "set-cookie:",
      NULL
   };

   unsigned int flags[] = {
      TYPE_FOUND,
      LENGTH_FOUND,
      DISPOSITION_FOUND,
      LANG_FOUND,
      CONTROL_FOUND,
      LOCATION_FOUND,
      COOKIE_FOUND,
      COOKIE2_FOUND,
      COOKIE3_FOUND,
      COOKIE4_FOUND,
      COOKIE5_FOUND
   };

   char **headers[] = {
      &item->content,
      &item->length,
      &item->disposition,
      &item->language,
      &item->control,
      &item->location,
      &item->cookie,
      &item->cookie2,
      &item->cookie3,
      &item->cookie4,
      &item->cookie5
   };

   downcase( item->buffer, 1 );

   for( sptr = strings, iptr = flags, hptr = headers;
        *sptr != NULL;
        ++sptr, ++iptr, ++hptr )
   {
      if (( item->flags & *iptr ) || *item->buffer != **sptr )
         continue;

      if (( match = extract_header( *sptr, item->buffer, &alloc_err )) == NULL )
      {
         if ( alloc_err )
         {
            free_conn( item );
            return 1;
         }

         continue;
      }

      item->flags |= *iptr;
      **hptr = match;
      break;
   }

   return 0;
}

int compose_response_header(  struct ccb *item )
{
   char *ptr;

   STRING_TRUNCATE( item->netstring )

   for( ptr = "HTTP/1."; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( string_append( item->netstring, ( item->flags & VERNO ? '1' : '0' )))
      return 1;

   ptr = ( item->flags & LOCATION_FOUND ? " 303 See Other\r\n" : " 200 OK\r\n" );

   for( ; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = server_header; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = date_header(); *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = "Content-Type: "; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   ptr = ( item->flags & TYPE_FOUND ? item->content : default_type );

   for( ; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = "\r\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->flags & LANG_FOUND )
   {
      for( ptr = "Content-Language: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->language; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   }
   else if ( content_language != NULL )
   {
      for( ptr = content_language; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & LOCATION_FOUND )
   {
      for( ptr = "Location: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->location; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & LENGTH_FOUND )
   {
      for( ptr = "Content-Length: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->length; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( ! ( item->flags & LOCATION_FOUND ) && item->next == CHUNK )
   {
      for( ptr = "Transfer-Encoding: chunked\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & CLOSE )
   {
      for( ptr = "Connection: close\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & DISPOSITION_FOUND )
   {
      for( ptr = "Content-Disposition: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->disposition; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & CONTROL_FOUND )
   {
      for( ptr = "Cache-Control: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->control; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & COOKIE_FOUND )
   {
      for( ptr = "Set-Cookie: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->cookie; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & COOKIE2_FOUND )
   {
      for( ptr = "Set-Cookie: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->cookie2; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & COOKIE3_FOUND )
   {
      for( ptr = "Set-Cookie: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->cookie3; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & COOKIE4_FOUND )
   {
      for( ptr = "Set-Cookie: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->cookie4; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   if ( item->flags & COOKIE5_FOUND )
   {
      for( ptr = "Set-Cookie: "; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = item->cookie5; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = "\r\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;
   }

   for( ptr = "\r\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   item->count = item->netstring->used;
   return 0;
}

void scgi_filter( struct ccb *item )
{
   int notready;

   item->start = time( NULL );

   for( ; ; )
   {
      notready = grab_scgi_line( item );

      /*
       * 0 is success.
       * 1 is premature EOF from CGI program or read error.
       * 2 is idle connection.
       * 3 is a too-long response header line.
       */

      switch( notready )
      {
         case 0:
            break;

         case 1:
         case 3:
            item->error = ERR_INTERNAL;
            item->state = ERROR;

            stop_read_source( item );
            flip_events( item );
            return;

         case 2:
            return;

         default:
            free_conn( item );
            return;
      }

      if ( *item->buffer == '\0' )
         break;

      if ( extract_scgi_headers( item ))
         return;
   }

   if ( item->flags & LOCATION_FOUND )
      item->next = NULL;
   else if ( ! ( item->flags & VERNO ) || ( item->flags & LENGTH_FOUND ))
      item->next = NOCHUNK;
   else
      item->next = CHUNK;

   item->state = HEADER;
   if ( compose_response_header( item ))
   {
      free_conn( item );
      return;
   }

   stop_read_source( item );
   flip_events( item );

   item->size = 1;
}

void reset_headers( struct ccb *item )
{
   if ( item->flags & TYPE_FOUND )
   {
      free( item->content );
      item->content = NULL;
      item->flags &= ~TYPE_FOUND;
   }

   if ( item->flags & COOKIE_FOUND )
   {
      free( item->cookie );
      item->cookie = NULL;
      item->flags &= ~COOKIE_FOUND;
   }

   if ( item->flags & LENGTH_FOUND )
   {
      free( item->length );
      item->length = NULL;
      item->flags &= ~LENGTH_FOUND;
   }

   if ( item->flags & LANG_FOUND )
   {
      free( item->language );
      item->language = NULL;
      item->flags &= ~LANG_FOUND;
   }
}

int read_body( struct ccb *item )
{
   stop_write_source( item );

   item->count = SSL_read( item->ssl, item->buffer, MIN( item->size, IOBUFSIZE ));

   if ( item->count <= 0 )
   {
      off_t size = item->size;
      int err = set_socket_events( item, item->count );

      switch( err )
      {
         case SSL_ERROR_WANT_READ:
         case SSL_ERROR_WANT_WRITE:
            item->count = 0;
            break;

         default:
            syslog( LOG_ERR, "read_body(): EOF: %ld bytes not delivered by client", size );
       }

      return 1;
   }

   item->written = 0;
   return 0;
}

void set_SCGI_body_events( struct ccb *item )
{
   if ( item->count || item->size )
   {
      start_write_source( item );
      return;
   }

   stop_write_source( item );
   stop_read_socket( item );
   stop_write_socket( item );
   add_read_source( item );

   reset_headers( item );
   item->start = time( NULL );
   item->state = SCGI_FILTER;
}

void scgi_body( struct ccb *item )
{
   off_t written;

   item->start = time( NULL );

   if ( ! item->count )
   {
      if ( read_body( item ))
         return;

      stop_read_socket( item );
      stop_write_socket( item );
   }

   written = write( item->source, &item->buffer[ item->written ], item->count );

   if ( written < 0 )
   {
      if ( errno == EWOULDBLOCK )
         start_write_source( item );
      else
      {
         syslog( LOG_ERR, "scgi_body(): write( item->source ): %m" );
         free_conn( item );
      }

      return;
   }

   item->count   -= written;
   item->written += written;
   item->size    -= written;

   set_SCGI_body_events( item );
}

void scgi_header( struct ccb *item )
{
   off_t written;

   item->start = time( NULL );

   written = write( item->source, &item->netstring->str[ item->written ], item->count );

   if ( written < 0 )
   {
      if ( errno != EWOULDBLOCK )
      {
         syslog( LOG_ERR, "scgi_header(): write( item->source ): %m" );
         free_conn( item );
      }

      return;
   }

   item->count   -= written;
   item->written += written;

   if ( ! item->count )
   {
      if ( item->size )
      {
         item->state = SCGI_BODY;
         return;
      }

      stop_write_source( item );
      add_read_source( item );

      reset_headers( item );
      item->state = SCGI_FILTER;
   }
}

int make_netstring( struct ccb *item )
{
   char *ptr, *addr, size[ 16 ];
   int len;

   if (( item->netstring = make_string()) == NULL )
      return 1;

   for( ptr = "CONTENT_LENGTH\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->length == NULL )
   {
      if ( string_append( item->netstring, '0' ))
         return 1;
   }
   else
      for( ptr = item->length; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "REQUEST_METHOD\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   ptr = ( item->type == 1 ? "GET\n" : "POST\n" );

   for( ; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = "QUERY_STRING\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = item->request; *ptr; ++ptr )
      if ( *ptr == '?' )
      {
         while( *++ptr )
            if ( string_append( item->netstring, *ptr ))
               return 1;

         break;
      }

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "CONTENT_TYPE\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->content != NULL )
      for( ptr = item->content; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_COOKIE\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->cookie != NULL )
      for( ptr = item->cookie; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_ACCEPT_LANGUAGE\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->language != NULL )
      for( ptr = item->language; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "SCRIPT_NAME\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   for( ptr = item->request; *ptr; ++ptr )
      if ( *ptr == '?' )
         break;
      else
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_REFERRER\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->referer != NULL )
      for( ptr = item->referer; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_ORIGIN\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->origin != NULL )
      for( ptr = item->origin; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_SEC_GPC\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->gpc != NULL )
      for( ptr = item->gpc; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_USER_AGENT\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->agent != NULL )
      for( ptr = item->agent; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_IF_MODIFIED_SINCE\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->since != NULL )
      for( ptr = item->since ; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   for( ptr = "HTTP_IF_UMODIFIED_SINCE\n"; *ptr; ++ptr )
      if ( string_append( item->netstring, *ptr ))
         return 1;

   if ( item->unmodified != NULL )
      for( ptr = item->unmodified; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

   if ( string_append( item->netstring, '\n' ))
      return 1;

   if (( addr = get_local_address( item )) != NULL )
   {
      for( ptr = "LOCAL_ADDR\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = addr; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      if ( string_append( item->netstring, '\n' ))
         return 1;
   }

   if (( addr = get_peer_address( item->sock, NULL, NULL )) != NULL )
   {
      for( ptr = "REMOTE_ADDR\n"; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      for( ptr = addr; *ptr; ++ptr )
         if ( string_append( item->netstring, *ptr ))
            return 1;

      if ( string_append( item->netstring, '\n' ))
         return 1;
   }

   len = snprintf( size, sizeof( size ), "%d:", item->netstring->used );

   if ( len < 0 || len >= sizeof( size ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "SCGI request header is too long. "
                          "Length must fit into 15 bytes when expressed as ASCII. "
                          "This should never happen.  Something is wrong with the universe." );

      return 1;
   }

   if ( string_append( item->netstring, ',' ))
      return 1;

   for( ptr = &size[ len - 1 ]; ptr >= size; --ptr )
      if ( string_prepend( item->netstring, *ptr ))
         return 1;

   item->size    = ( item->flags & LENGTH_FOUND ? strtol( item->length, NULL, 10 ) : 0 );
   item->count   = item->netstring->used;
   item->written = 0;

   for( ptr = item->netstring->str; *ptr; ++ptr )
      if ( *ptr == '\n' )
         *ptr = '\0';

   return 0;
}

void scgi_prepare( struct ccb *item )
{
   int err = 0;
   socklen_t slen = sizeof( err );

   start_write_socket( item );
   stop_write_source( item );

   if ( getsockopt( item->source, SOL_SOCKET, SO_ERROR, &err, &slen ) < 0 )
   {
      syslog( LOG_ERR, "getsockopt( item->source ): %m" );
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( err )
   {
      syslog( LOG_ERR, "connect(): %s", strerror( err ));
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( make_netstring( item ))
   {
      free_conn( item );
      return;
   }

   stop_write_socket( item );
   start_write_source( item );
   item->state = SCGI_HEADER;
}

void scgi_connect( struct ccb *item )
{
   struct sockaddr_un sa;
   int fd = -1;

   if ( *item->scgi != '/' )
   {
      if (( fd = connect_to_server( item->scgi )) > 0 )
         goto CONTINUE;

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if (( fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   non_blocking( fd );

   bzero( &sa, sizeof( struct sockaddr_un ));
   sa.sun_family = AF_LOCAL;
   strncpy( sa.sun_path, item->scgi, sizeof( sa.sun_path ) - 1 );

   if ( connect( fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )) < 0 )
   {
      if ( errno != EINPROGRESS )
      {
         syslog( LOG_ERR, "connect(): %m" );
         close( fd );

         item->state = ERROR;
         item->error = ERR_INTERNAL;
         return;
      }
   }

CONTINUE:
   item->source = fd;
   item->state = SCGI_PREPARE;

   stop_write_socket( item );
   add_write_source( item );
}

int is_scgi( struct ccb *item )
{
   char *ptr1, *ptr2, **pptr, **sptr, *paths[] = { SCGI1, SCGI2, SCGI3, SCGI4, SCGI5, NULL },
                                      *servs[] = { scgi1, scgi2, scgi3, scgi4, scgi5 };

   item->scgi = NULL;

   for( pptr = paths, sptr = servs; *pptr != NULL; ++pptr, ++sptr )
   {
      ptr1 = *pptr;
      ptr2 = item->request;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 )
      {
         item->scgi = *sptr;
         break;
      }
   }

   if ( item->scgi == NULL || ! *item->scgi )
      return 0;

   if ( ! item->type )
   {
      item->state = ERROR;
      item->error = ERR_BADREQUEST;
      return 1;
   }

   if ( scgi_cross_origin && reject_cross_origin( item, 0 ))
      return 1;

   if ( item->flags & EXPECT_FOUND )
   {
      downcase( item->expect, 0 );

      ptr1 = "100-continue";
      ptr2 = item->expect;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 && ! *ptr2 )
      {
         if (( item->count = snprintf( item->buffer, IOBUFSIZE, "HTTP/1.%d 100 Continue\r\n\r\n", ( item->flags & VERNO ? 1 : 0 ))) < 0 )
         {
            syslog( LOG_ERR, "snprintf(): %m" );
            free_conn( item );
            return 1;
         }

         item->state = HEADER;
         item->next = SCGI_CONNECT;
         return 1;
      }
   }

   scgi_connect( item );
   return 1;
}

void in_socket( struct ccb *item )
{
   if (( item->sock_count = SSL_read( item->ssl, item->buffer, HALFSIZE )) <= 0 )
   {
      int err = set_socket_events( item, item->sock_count );

      if ( err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE )
      {
         clear_events( item );
         stop_read_source( item );
         item->sock_opposite = ( err == SSL_ERROR_WANT_READ ? 0 : 1 );
      }

      return;
   }

   item->sock_opposite  = 0;
   item->source_written = 0;

   if ( item->source_count )
      start_write_socket( item );
   else
   {
      stop_write_socket( item );
      start_read_source( item );
   }

   stop_read_socket( item );
   start_write_source( item );
}

void out_source( struct ccb *item )
{
   int written;

   if (( written = write( item->source, &item->buffer[ item->source_written ], item->sock_count )) <= 0 )
   {
      free_conn( item );
      return;
   }

   item->source_written += written;
   item->sock_count     -= written;

   if ( ! item->sock_count )
   {
      stop_write_source( item );
      start_read_socket( item );

      if ( SSL_has_pending( item->ssl ))
         in_socket( item );
   }
}

void in_source( struct ccb *item )
{
   if (( item->source_count = read( item->source, &item->buffer[ HALFSIZE ], HALFSIZE )) <= 0 )
   {
      free_conn( item );
      return;
   }

   stop_read_source( item );
   start_write_socket( item );
}

void out_socket( struct ccb *item )
{
   int written;

   if (( written = SSL_write( item->ssl, &item->buffer[ HALFSIZE ], item->source_count )) <= 0 )
   {
      int err = set_socket_events( item, written );

      if ( err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE )
      {
         clear_events( item );
         stop_write_source( item );
         item->sock_opposite = ( err == SSL_ERROR_WANT_WRITE ? 0 : 1 );
      }

      return;
   }

   item->sock_opposite = 0;
   item->source_count  = 0;

   if ( ! item->sock_count )
      start_read_socket( item );
   else
   {
      stop_read_socket( item );
      start_write_source( item );
   }

   stop_write_socket( item );
   start_read_source( item );
}

void websocket_transfer( struct ccb *item )
{
   if ( item->fd == item->source )
   {
      if ( item->filter == EVFILT_READ )
         in_source( item );
      else
         out_source( item );

      return;
   }

   if ( item->filter == EVFILT_READ )
   {
      if ( item->sock_opposite )
         out_socket( item );
      else
         in_socket( item );

      return;
   }

   if ( item->sock_opposite )
      in_socket( item );
   else
      out_socket( item );
}

/*
 * Do not deliver HTTP error messages.  Connection has been upgraded.  It
 * is no longer an HTTP connection.
 */

void websocket_verify( struct ccb *item )
{
   int err = 0;
   socklen_t slen = sizeof( err );

   if ( getsockopt( item->source, SOL_SOCKET, SO_ERROR, &err, &slen ) < 0 )
   {
      syslog( LOG_ERR, "getsockopt( item->source ): %m" );
      free_conn( item );
      return;
   }

   if ( err )
   {
      syslog( LOG_ERR, "connect(): %s", strerror( err ));
      free_conn( item );
      return;
   }

   add_read_source( item );

   item->state = WS_TRANSFER;
   item->sock_count = snprintf( item->buffer, HALFSIZE, "%s\n", (( item->flags & COOKIE_FOUND ) ? item->cookie : "" ));

   --active;

   if ( ! active )
      delete_timer();

   if ( item->sock_count < 0 || item->sock_count >= HALFSIZE )
   {
      if ( item->sock_count < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized cookie header for a WebSocket server. "
                          "Dropping connection.  See -b option in manual." );
      free_conn( item );
      return;
   }
}

int connect_to_server( char *destination )
{
   int result, fd;
   char *port;
   struct addrinfo hints, *res, *ptr;

   for( port = destination; *port; ++port )
      ;

   ++port;

   bzero( &hints, sizeof( struct addrinfo ));
   hints.ai_family = PF_UNSPEC;
   hints.ai_socktype = SOCK_STREAM;

   if (( result = getaddrinfo( destination, port, &hints, &res )))
   {
      syslog( LOG_ERR, "getaddrinfo(): %s", gai_strerror( result ));
      return -1;
   }

   for( ptr = res; ptr != NULL; ptr = ptr->ai_next )
   {
      result = 0;
      fd = socket( ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol );

      if ( fd == -1 )
         continue;

      non_blocking( fd );

      if (( result = connect( fd, ptr->ai_addr, ptr->ai_addrlen )) == -1 )
      {
         if ( errno != EINPROGRESS )
         {
            close( fd );
            fd = -1;
            continue;
         }
      }

      break;
   }

   if ( fd == -1 )
      syslog( LOG_ERR, ( result ? "connect(): %m" : "socket(): %m" ));

   if ( res != NULL )
      freeaddrinfo( res );

   return fd;
}

void websocket_connect( struct ccb *item )
{
   struct sockaddr_un sa;
   int fd = -1;

   if ( *item->websocket != '/' )
   {
      if (( fd = connect_to_server( item->websocket )) > 0 )
         goto CONTINUE;

      free_conn( item );
      return;
   }

   if (( fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      free_conn( item );
      return;
   }

   non_blocking( fd );

   bzero( &sa, sizeof( struct sockaddr_un ));
   sa.sun_family = AF_LOCAL;
   strncpy( sa.sun_path, item->websocket, sizeof( sa.sun_path ) - 1 );

   if ( connect( fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )) < 0 )
   {
      if ( errno != EINPROGRESS )
      {
         syslog( LOG_ERR, "connect(): %m" );
         close( fd );
         free_conn( item );
         return;
      }
   }

CONTINUE:
   item->source = fd;
   item->state = WS_VERIFY;

   stop_write_socket( item );
   add_write_source( item );
}

char *base64_encode( char *input, int len )
{
   int pad, i;
   char *ptr, buff[ 3 ], *trailer;
   static struct string *s = NULL;
   static char *encs = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                       "abcdefghijklmnopqrstuvwxyz"
                       "0123456789+/";

   if ( s == NULL )
   {
      if (( s = make_string()) == NULL )
         return NULL;
   }
   else
   {
      s->free += s->used;
      s->used = 0;
      s->top = s->str;
   }

   pad = len % 3;
   ptr = input;

   if ( len > 2 )
   {
      len -= pad;

      for( i = 0; i < len; i += 3 )
      {
         buff[ 0 ] = *ptr++;
         buff[ 1 ] = *ptr++;
         buff[ 2 ] = *ptr++;

         if ( string_append( s, encs[ ( buff[ 0 ] & 0xfc ) >> 2 ] ))
            return NULL;

         if ( string_append( s, encs[ (( buff[ 0 ] & 0x03 ) << 4 ) | (( buff[ 1 ] & 0xf0 ) >> 4 ) ] ))
            return NULL ;

         if ( string_append( s, encs[ (( buff[ 1 ] & 0x0f ) << 2 ) | (( buff[ 2 ] & 0xc0 ) >> 6 ) ] ))
            return NULL;

         if ( string_append( s, encs[ buff[ 2 ] & 0x3f ] ))
            return NULL;
      }
   }
   else
      pad = len;

   if ( pad )
   {
      buff[ 0 ] = *ptr++;

      if ( string_append( s, encs[  ( buff[ 0 ] & 0xfc ) >> 2 ] ))
         return NULL;

      if ( --pad )
      {
         trailer = "=";
         buff[ 1 ] = *ptr;

         if ( string_append( s, encs[ (( buff[ 0 ] & 0x03 ) << 4 ) | (( buff[ 1 ] & 0xf0 ) >> 4 ) ] ))
            return NULL;

         if ( string_append( s, encs[ (( buff[ 1 ] & 0x0f ) << 2 ) ] ))
            return NULL;
      }
      else
      {
         trailer = "==";
         if ( string_append( s, encs[ (( buff[ 0 ] & 0x03 ) << 4 ) ] ))
            return NULL;
      }

      for( ptr = trailer; *ptr; ++ptr )
         if ( string_append( s, *ptr ))
            return NULL;
   }

   return s->str;
}

void websocket_handshake( struct ccb *item )
{
   unsigned int len;
   char buffer[ 1024 ], *ptr1, *ptr2, **pptr, **sptr,
        *paths[] = { WEBSOCKET1, WEBSOCKET2, WEBSOCKET3, WEBSOCKET4, WEBSOCKET5, NULL },
        *servs[] = { websocket1, websocket2, websocket3, websocket5, websocket5 };

   unsigned char *hashed;
   EVP_MD_CTX *ctx;

   item->websocket = NULL;

   for( pptr = paths, sptr = servs; *pptr != NULL; ++pptr, ++sptr )
   {
      ptr1 = *pptr;
      ptr2 = item->request;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 )
      {
         item->websocket = *sptr;
         break;
      }
   }

   if ( item->websocket == NULL || ! *item->websocket )
   {
      item->state = ERROR;
      item->error = ERR_NOTFOUND;
      return;
   }

   if ( item->origin == NULL )
   {
      item->state = ERROR;
      item->error = ERR_BADREQUEST;
      return;
   }

   if ( ws_cross_origin && reject_cross_origin( item, 0 ))
      return;

   if ( item->type != 1 || ! ( item->flags & VERNO ) || item->ws_key == NULL || item->ws_verno == NULL )
   {
      item->state = ERROR;
      item->error = ERR_BADREQUEST;
      return;
   }

   if ( strtol( item->ws_verno, NULL, 10 ) != 13 )
   {
      item->state = ERROR;
      item->error = ERR_WSVERSION;
      return;
   }

   len = snprintf( buffer, sizeof( buffer ), "%s%s", item->ws_key, ws_guid );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      if ( len < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized Websocket Accept key. "
                          "Client is malicious!  Dropping connection." );
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if (( ctx = EVP_MD_CTX_create()) == NULL )
   {
      log_error();
      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( ! EVP_DigestInit_ex( ctx, EVP_sha1(), NULL ))
   {
      log_error();
      EVP_MD_CTX_destroy( ctx );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( ! EVP_DigestUpdate( ctx, buffer, len ))
   {
      log_error();
      EVP_MD_CTX_destroy( ctx );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if (( hashed = OPENSSL_malloc( EVP_MD_size( EVP_sha1() ))) == NULL )
   {
      log_error();
      EVP_MD_CTX_destroy( ctx );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   if ( ! EVP_DigestFinal_ex( ctx, hashed, &len ))
   {
      log_error();
      EVP_MD_CTX_destroy( ctx );
      OPENSSL_free( hashed );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   EVP_MD_CTX_destroy( ctx );

   if (( ptr1 = base64_encode(( char *)hashed, len )) == NULL )
   {
      OPENSSL_free( hashed );
      free_conn( item );
      return;
   }

   OPENSSL_free( hashed );

   item->count = snprintf( item->buffer, IOBUFSIZE,
                           "HTTP/1.1 101 Switching Protocols\r\n"
                           "Connection: Upgrade\r\n"
                           "Upgrade: websocket\r\n"
                           "Sec-WebSocket-Accept: %s\r\n"
                           "\r\n",
                           ptr1 );

   if ( item->count < 0 || item->count >= IOBUFSIZE )
   {
      if ( item->count < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized WebSocket handshake response. "
                          "This shouldn't happen.  Dropping connection." );

      item->state = ERROR;
      item->error = ERR_INTERNAL;
      return;
   }

   item->state = HEADER;
   item->next = WS_CONNECT;
   return;
}

int get_connection_flags( struct ccb *item )
{
   int upgrade = 0;
   char *ptr1, *ptr2, *ptr3, *ptr4, **sptr, *strings[] = { "close", "upgrade", NULL };

   sptr = strings;
   ptr1 = item->connection;

   while(( ptr2 = strsep( &ptr1, " ,;" )) != NULL )
   {
      if ( ! *ptr2 )
         continue;

      for( sptr = strings; *sptr != NULL; ++sptr )
      {
         ptr3 = *sptr;
         ptr4 = ptr2;

         while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
         {
            ++ptr3;
            ++ptr4;
         }

         if ( ! *ptr3 && ! *ptr4 )
         {
            if ( *sptr == strings[ 0 ] )
               item->flags |= CLOSE;
            else
               ++upgrade;
         }
      }
   }

   return upgrade;
}

int is_websocket( struct ccb *item )
{
   char *ptr1, *ptr2;

   if ( item->connection == NULL )
      return 0;

   downcase( item->connection, 0 );

   if ( ! get_connection_flags( item ) || item->upgrade == NULL )
      return 0;

   downcase( item->upgrade, 0 );

   ptr1 = item->upgrade;
   ptr2 = "websocket";

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
   {
      websocket_handshake( item );
      return 1;
   }

   return 0;
}

int get_method( struct ccb *item )
{
   char *ptr1, *ptr2;
   static char *head = "HEAD", *get = "GET", *post = "POST";

   ptr1 = item->method;
   ptr2 = head;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   item->type = -1;

   if ( ! *ptr1 && ! *ptr2 )
   {
      item->type = 0;
      return 0;
   }

   ptr1 = item->method;
   ptr2 = get;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
   {
      item->type = 1;
      return 0;
   }

   ptr1 = item->method;
   ptr2 = post;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
      item->type = 2;

   if ( item->type < 0 )
   {
      item->error = ERR_BADMETHOD;
      item->state = ERROR;
      return 1;
   }

   return 0;
}

int get_version( struct ccb *item )
{
   char *ptr1, *ptr2;
   static char *http = "HTTP/";

   ptr1 = item->version;
   ptr2 = http;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( *ptr2 )
   {
      item->error = ERR_BADREQUEST;
      item->state = ERROR;
      return 1;
   }

   if ( ! *ptr1
        || *ptr1 != '1'
        || *( ptr1 + 1 ) != '.'
        || ( *( ptr1 + 2 ) != '0' && *( ptr1 + 2 ) != '1' )
        || *( ptr1 + 3 ))
   {
      item->error = ERR_BADVERSION;
      item->state = ERROR;
      return 1;
   }

   if ( *( ptr1 + 2 ) == '1' )
      item->flags |= VERNO;
   else
      item->flags |= CLOSE;

   return 0;
}

int check_for_forbidden( struct ccb *item )
{
   char *ptr;

   if ( item->request[ 0 ] == '.' && item->request[ 1 ] == '.' && item->request[ 2 ] == '/' )
   {
      item->error = ERR_BADREQUEST;
      item->state = ERROR;
      return 1;
   }

   for( ptr = item->request; *ptr; ++ptr )
      if ( *ptr == '/' && *( ptr + 1 ) == '.' && *( ptr + 2 ) == '.' && ( ! *( ptr + 3 ) || *( ptr + 3 ) == '/' ))
      {
         item->error = ERR_BADREQUEST;
         item->state = ERROR;
         return 1;
      }

   return 0;
}

int substitute_root_resource( struct ccb *item )
{
   int len;
   char *ptr;

   for( len = 0, ptr = item->request; *ptr; ++ptr )
      ++len;

   if ( item->request[ len - 1 ] == '/' )
   {
      char *ptr;

      len += 11;

      if (( ptr = memory( len )) == NULL )
      {
         free_conn( item );
         return 1;
      }

      if ( snprintf( ptr, len, "%s%s", item->request, root_resource ) < 0 )
      {
         syslog( LOG_ERR, "snprintf(): %m" );
         free( ptr );
         free_conn( item );
         return 1;
      }

      free( item->request );
      item->request = ptr;
   }

   return 0;
}

void percent_decode( struct ccb *item )
{
   char *ptr, *ptr1, *ptr2;

   for( ptr = item->request; *ptr; ++ptr )
   {
      char c1, c2;

      if ( *ptr == '%' && *( ptr + 1 ) && *( ptr + 2 ))
      {
         c1 = *( ptr + 1 );
         c2 = *( ptr + 2 );

         if ((( c1 < '0' || c1 > '9' ) && ( c1 < 'a' || c1 > 'f' ) && ( c1 < 'A' || c1 > 'F' )) ||
             (( c2 < '0' || c2 > '9' ) && ( c2 < 'a' || c2 > 'f' ) && ( c2 < 'A' || c2 > 'F' )))
            continue;

         if ( c1 >= 'A' && c1 <= 'F' )
            c1 += 32;

         if ( c2 >= 'A' && c2 <= 'F' )
            c2 += 32;

         if ( c1 >= 'a' && c1 <= 'f' )
            c1 = ( c1 - 87 ) * 16;
         else
            c1 = ( c1 - 48 ) * 16;

         if ( c2 >= 'a' && c2 <= 'f' )
            c2 -= 87;
         else
            c2 -= 48;

         *ptr = c1 + c2;

         for( ptr1 = ptr + 1, ptr2 = ptr + 3; *ptr2; ++ptr1, ++ptr2 )
            *ptr1 = *ptr2;

         *ptr1 = '\0';
      }
   }
}

int log_request( struct ccb *item )
{
   char *ptr, *addr, buffer[ MAXPATHLEN ], port[ 16 ];
   time_t t;
   struct tm *lt;
   int portnum, ipv6;

   if ( logfile == NULL )
      return 0;

   if (( addr = get_peer_address( item->sock, &portnum, &ipv6 )) == NULL )
      return 1;

   t = time( NULL );
   lt = gmtime( &t );
   strftime( buffer, sizeof( buffer ), "%Y-%m-%d %H:%M:%S ", lt );

   STRING_TRUNCATE( logline )

   for( ptr = buffer; *ptr; ++ptr )
      if ( string_append( logline, *ptr ))
         return 1;

   if ( ipv6 && string_append( logline, '[' ))
      return 1;

   for( ptr = addr; *ptr; ++ptr )
      if ( string_append( logline, *ptr ))
         return 1;

   if ( ipv6 && string_append( logline, ']' ))
      return 1;

   snprintf( port, sizeof( port ), ":%d", portnum );

   for( ptr = port; *ptr; ++ptr )
      if ( string_append( logline, *ptr ))
         return 1;

   if ( string_append( logline, ' ' ))
      return 1;

   if ( string_append( logline, '"' ))
      return 1;

   if ( item->agent != NULL )
      for( ptr = item->agent; *ptr; ++ptr )
      {
         if ( string_append( logline, *ptr ))
            return 1;

         if ( *ptr == '"' && string_append( logline, '"' ))
            return 1;
      }

   if ( string_append( logline, '"' ))
      return 1;

   if ( string_append( logline, ' ' ))
      return 1;

   if ( string_append( logline, '"' ))
      return 1;

   if ( item->origin != NULL )
      for( ptr = item->origin; *ptr; ++ptr )
      {
         if ( string_append( logline, *ptr ))
            return 1;

         if ( *ptr == '"' && string_append( logline, '"' ))
            return 1;
      }

   if ( string_append( logline, '"' ))
      return 1;

   if ( string_append( logline, ' ' ))
      return 1;

   if ( string_append( logline, '"' ))
      return 1;

   if ( item->referer != NULL )
      for( ptr = item->referer; *ptr; ++ptr )
      {
         if ( string_append( logline, *ptr ))
            return 1;

         if ( *ptr == '"' && string_append( logline, '"' ))
            return 1;
      }

   if ( string_append( logline, '"' ))
      return 1;

   if ( string_append( logline, ' ' ))
      return 1;

   for( ptr = item->method; *ptr; ++ptr )
      if ( string_append( logline, *ptr ))
         return 1;

   if ( string_append( logline, ' ' ))
      return 1;

   for( ptr = item->request; *ptr; ++ptr )
      if ( string_append( logline, *ptr ))
         return 1;

   if ( string_append( logline, '\n' ))
      return 1;

   if ( fputs( logline->str, logfile ) == EOF )
   {
      syslog( LOG_ERR, "fputs( logline ): %m" );
      fclose( logfile );
      logfile = NULL;
      logging = 0;
   }

   return 0;
}

void analyze_request( struct ccb *item )
{
   if ( logging && log_request( item ))
   {
      free_conn( item );
      return;
   }

   if ( get_method( item ) || get_version( item ))
      return;

   percent_decode( item );

   if ( reject_other_hostnames( item ))
      return;

   if ( check_for_forbidden( item ))
      return;

   if ( is_websocket( item ))
      return;

   if ( is_scgi( item ))
      return;

   if ( substitute_root_resource( item ))
      return;

   prepare_open( item );
}

void respond_header( struct ccb *item )
{
   off_t written;
   char *ptr;

   item->start = time( NULL );

   if ( item->netstring != NULL )
      ptr = item->netstring->str;
   else
      ptr = item->buffer;

   if (( written = SSL_write( item->ssl, ptr, item->count )) <= 0 )
   {
      set_socket_events( item, written );
      return;
   }

   if (( item->state = item->next ) == NULL )
   {
      (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
      return;
   }

   if ( item->state == RESPOND || item->state == RESPOND_KTLS )
   {
      if ( ! item->size )
      {
         (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
         return;
      }
   }

   item->count = 0;
   item->state( item );
}

int make_error( struct ccb *item )
{
   char buffer[ 2048 ], len_buffer[ 64 ];
   int count;

   if ( item->error == ERR_NOTMODIFIED || item->error == ERR_MODIFIED  || item->error == ERR_RANGE )
      count = 0;
   else
      count = snprintf( buffer, sizeof( buffer ),
                   "<!DOCTYPE html><html lang=\"en\"><head><title>Error</title></head><body><h1>%s</h1></body></html>",
                   errstrings[ item->error ] );

   if ( count < 0 || count >= sizeof( buffer ))
   {
      if ( count < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized error response. "
                          "This should NEVER happen!  Dropping connection." );
      free_conn( item );
      return 1;
   }

   if ( snprintf( len_buffer, sizeof( len_buffer ), "Content-Length: %u\r\n", count ) < 0 )
   {
      syslog( LOG_ERR, "snprintf(): %m" );
      free_conn( item );
      return 1;
   }

   item->count = snprintf( item->buffer, IOBUFSIZE, "HTTP/1.%d%s"
                           "%s"
                           "%s"
                           "%s"
                           "%s"
                           "%s"
                           "\r\n"
                           "%s",

                           ( item->flags & CLOSE ? 0 : 1 ), errstrings[ item->error ],
                           ( count ? "Content-Type: text/html\r\n" : "" ),

                           server_header,
                           date_header(),

                           ( count ? len_buffer : "" ),
                           ( item->flags & CLOSE ? "Connection: close\r\n" : "" ),
                           ( count ? buffer : "" ));


   if ( item->count < 0 || item->count >= IOBUFSIZE )
   {
      if ( item->count < 0 )
         syslog( LOG_ERR, "snprintf(): %m" );
      else
         syslog( LOG_ERR, "Prospero generated a oversized error response. "
                          "This should NEVER happen!  Dropping connection." );
      free_conn( item );
      return 1;
   }

   return 0;
}

void respond_error( struct ccb *item )
{
   off_t written;

   if ( ! item->count && make_error( item ))
      return;

   item->start = time( NULL );

   if (( written = SSL_write( item->ssl, item->buffer, item->count )) <= 0 )
   {
      set_socket_events( item, written );
      return;
   }

   (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
}

void respond_static_ktls( struct ccb *item )
{
   ssize_t written;

   item->start = time( NULL );

   written = SSL_sendfile( item->ssl, item->source, item->written, MIN( item->size, IOBUFSIZE ), 0 );

   if ( written < 0 )
   {
      set_socket_events( item, written );
      return;
   }

   if ( ! ( item->size -= written ))
   {
      (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
      return;
   }

   item->written += written;
   flip_events( item );
}

void respond_static( struct ccb *item )
{
   off_t written = 0;
   int result;

   item->start = time( NULL );

   if ( ! item->count )
   {
      switch(( result = read( item->source, item->buffer, MIN( item->size, IOBUFSIZE ))))
      {
         case 0:
         case -1:
            free_conn( item );
            return;

         default:
            item->count = result;
      }
   }

   if (( written = SSL_write( item->ssl, item->buffer, item->count )) <= 0 )
   {
      set_socket_events( item, written );
      return;
   }

   if ( ! ( item->size -= written ))
   {
      (( item->flags & CLOSE ) ? free_conn( item ) : clear_conn( item ));
      return;
   }

   item->count = 0;
   flip_events( item );
}

int grab_insecure_line( struct ccb *item )
{
   int r;
   char c;

   if ( ! ( item->flags & INACTIVE ))
   {
      item->count = 0;
      *item->buffer = '\0';
   }

   for( ; ; )
   {
      r = read( item->sock, &c, 1 );

      if ( r <= 0 )
      {
         if ( r < 0 && errno == EWOULDBLOCK )
         {
            item->flags |= INACTIVE;
            return 2;
         }

         return 1;
      }

      /*
       * We don't want the line terminators.
       */

      if ( c == 13 )
         continue;

      if ( c == 10 )
         break;

      /*
       * New character overwrites previous string terminator.  New string
       * terminator added.
       */

      item->buffer[ item->count++ ] = c;

      if ( item->count >= IOBUFSIZE )
         return 3;

      item->buffer[ item->count ] = '\0';
   }

   item->flags &= ~INACTIVE;
   return 0;
}

void redirect_insecure_request( struct ccb *item )
{
   static char *template = "HTTP/1.0 301 Moved Permanently\r\nLocation: %s%s\r\n\r\n";
   char buffer[ MAXPATHLEN * 2 ];
   int len, notready;

   item->start = time( NULL );
   notready = grab_insecure_line( item );

   /*
    * 0 is success.
    * 1 is premature EOF or read error.
    * 2 is idle connection.
    * 3 is a too-long request header line.
    */

   switch( notready )
   {
      case 0:
         break;

      case 2:
         return;

      default:
         free_conn( item );
         return;
   }

   if ( grab_request( item ) || get_method( item ) || get_version( item ))
   {
      if ( item->error )
      {
         item->flags |= CLOSE;

         if ( ! make_error( item ))
            write( item->sock, buffer, item->count );
      }

      free_conn( item );
      return;
   }

   if (( len = snprintf( buffer, sizeof( buffer ), template, hostname, item->request )) < 0 || len >= sizeof( buffer ))
   {
      free_conn( item );
      return;
   }

   write( item->sock, buffer, len );
   free_conn( item );
}

void accept_connection( int fd )
{
   int conn, new;

   new = 0;

   if ( fd < 0 )
      return;

   do
   {
      if ( sigterm )
         return;

      conn = accept( fd , NULL, NULL );

      if ( conn < 0 )
      {
         if ( errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
            syslog( LOG_ERR, "accept(): %m" );

         return;
      }

      add_conn( conn, ( fd == sec_fd ? 1 : 0 ));

   } while ( ++new <= NEWCONS );
}

void set_signals_intr()
{
   struct sigaction sigact;

   sigact.sa_handler = sigterm_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGTERM, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction( SIGTERM ): %m" );

   sigact.sa_handler = sighup_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGHUP, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction( SIGHUP ): %m" );
}

void timeout_conns()
{
   int n;
   struct ccb **cptr;
   time_t now;

   now = time( NULL );

   for( n = 0, cptr = conns; n < max_conn; ++n, ++cptr )
   {
      if ( *cptr == NULL )
         continue;

      if (( *cptr )->state != WS_TRANSFER && ( now - ( *cptr )->start ) > TIMEOUT )
         free_conn( *cptr );
   }
}

void process_clients()
{
   int kq;
   struct ccb *item;
   struct kevent *eptr;

   qlen = max_conn * MAXEVENT;

   if (( conns = memory( sizeof( struct ccb * ) * max_conn )) == NULL )
      exit( 1 );

   bzero( conns, sizeof( struct ccb * ) * max_conn );

   if (( inqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( inqueue, sizeof( struct kevent ) * qlen );

   if (( outqueue = memory( sizeof( struct kevent ) * qlen )) == NULL )
      exit( 1 );

   bzero( outqueue, sizeof( struct kevent ) * qlen );

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   ev_set( insec_fd, EVFILT_READ, EV_ADD, NULL );
   ev_set( sec_fd, EVFILT_READ, EV_ADD, NULL );

   for( ; ; )
   {
      if ( sigterm )
         exit( 0 );

      if ( sighup )
         open_logfile();

      set_signals_intr();
      out = kevent( kq, inqueue, in, outqueue, qlen, NULL );
      in = 0;

      if ( out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %s", strerror( errno ));
         exit( 1 );
      }

      signal( SIGTERM, sigterm_handler );
      signal( SIGHUP, sighup_handler );

      for( idx = 0, eptr = outqueue; idx < out; ++idx, ++eptr )
      {
         if ( sigterm )
            exit( 0 );

         if ( sighup )
            open_logfile();

         if ( ! eptr->ident || eptr->flags & EV_ERROR )
            continue;

         if ( eptr->filter == EVFILT_TIMER )
         {
            timeout_conns();
            continue;
         }

         if ( eptr->ident == sec_fd || eptr->ident == insec_fd )
         {
            accept_connection( eptr->ident );
            continue;
         }

         if (( item = eptr->udata ) != NULL && item->state != NULL )
         {
            item->fd = eptr->ident;
            item->filter = eptr->filter;
            item->state( item );
            continue;
         }

         syslog( LOG_ERR, "total: %d, active: %d, idx: %d, out: %d, "
                          "item: %s, item->state: %s",
                 total, active, idx, out,
                 ( item == NULL ? "NULL" : "NON-NULL" ),
                 ( item != NULL && item->state == NULL ? "NULL" :
                 ( item == NULL ? "NA" : "NON-NULL" )));
      }
   }
}

char *get_line( int fd, int config )
{
   static struct string *s = NULL;
   static char buffer[ 128 ] = { 0 }, *ptr = buffer;
   int count;

   if ( s != NULL )
   {
      STRING_TRUNCATE( s )
   }
   else
   {
      if (( s = make_string()) == NULL )
         exit( 1 );
   }

   if ( fd < 0 )
   {
      *buffer = '\0';
      ptr = buffer;
      return NULL;
   }

AGAIN:
   if ( ! *ptr )
   {
      if (( count = read( fd, buffer, sizeof( buffer ) - 1 )) <= 0 )
      {
         if ( count < 0 )
         {
            syslog( LOG_ERR, "read of %s file %s failed: %s",
                     ( config ? "config" : "type" ), ( config ? config_file : type_file ),
                     strerror( errno ));
            exit( 1 );
         }

         if ( s->used )
            return ( char *)s->str;

         return NULL;
      }

      buffer[ count ] = '\0';
      ptr = buffer;
   }

   while( *ptr && *ptr != '\n' && *ptr != '\r' )
      if ( string_append( s, *ptr++ ))
         return NULL;

   if ( ! *ptr )
      goto AGAIN;

   if ( config )
   {
      if ( *ptr == '\r' )
         ++ptr;

      if ( *ptr == '\n' )
         ++ptr;
   }
   else
   {
      while( *ptr == '\n' || *ptr == '\r' )
         ++ptr;

      if ( ! s->used )
         goto AGAIN;
   }

   return ( char *)s->str;
}

void make_names( char *host )
{
   char buffer[ MAXPATHLEN ];
   int len;

   len = snprintf( buffer, sizeof( buffer ), "https://%s:443", host );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      syslog( LOG_ERR, ( len < 0 ? "snprintf(): %m" : "hostname is too long" ));
      exit( 1 );
   }

   if (( hostname_443 = str_dup( buffer, len )) == NULL )
   {
      syslog( LOG_ERR, "could not allocate hostname_443" );
      exit( 1 );
   }

   len = snprintf( buffer, sizeof( buffer ), "https://www.%s:443", host );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      syslog( LOG_ERR, ( len < 0 ? "snprintf(): %m" : "hostname is too long" ));
      exit( 1 );
   }

   if (( www_hostname_443 = str_dup( buffer, len )) == NULL )
   {
      syslog( LOG_ERR, "could not allocate www_hostname_443" );
      exit( 1 );
   }

   len = snprintf( buffer, sizeof( buffer ), "https://%s", host );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      syslog( LOG_ERR, ( len < 0 ? "snprintf(): %m" : "hostname is too long" ));
      exit( 1 );
   }

   if (( hostname = str_dup( buffer, len )) == NULL )
   {
      syslog( LOG_ERR, "could not allocate hostname" );
      exit( 1 );
   }

   len = snprintf( buffer, sizeof( buffer ), "https://www.%s", host );

   if ( len < 0 || len >= sizeof( buffer ))
   {
      syslog( LOG_ERR, ( len < 0 ? "snprintf(): %m" : "hostname is too long" ));
      exit( 1 );
   }

   if (( www_hostname = str_dup( buffer, len )) == NULL )
   {
      syslog( LOG_ERR, "could not allocate www_hostname" );
      exit( 1 );
   }
}

char *set_hostname( char *config )
{
   long len;
   char *ptr1, *ptr2, *host;

   if ( config != NULL && *config )
   {
      if (( host =str_dup( config, -1 )) == NULL )
         exit( 1 );

      return host;
   }

   if (( len = sysconf( _SC_HOST_NAME_MAX )) < 0 )
      len = MAXPATHLEN;

   ++len;
   if (( host = memory( len )) == NULL )
      exit( 1 );

   if ( gethostname( host, len ))
   {
      syslog( LOG_ERR, "gethostname(): %m" );
      exit( 1 );
   }

   ptr1 = "localhost";
   ptr2 = host;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 && ! *ptr2 )
   {
      ptr1 = host;
      ptr2 = "127.0.0.1";
      --len;

      while( len && *ptr2 )
      {
         *ptr1++ = *ptr2++;
         --len;
      }

      *ptr1 = '\0';
   }

   return host;
}

int verify_servers()
{
   char **ptr, *ptr1, *servers[] = { websocket1, scgi1, websocket2, scgi2, websocket3, scgi3,
                                     websocket4, scgi4, websocket5, scgi5, NULL };

   for( ptr = servers; *ptr != NULL; ++ptr )
   {
      if ( *ptr != NULL && **ptr && **ptr != '/' )
      {
         int found;

         for( ptr1 = *ptr; *ptr1; ++ptr1 )
            ;

         for( found = 0, --ptr1; ptr1 >= *ptr; --ptr1 )
            if ( *ptr1 == ':' && *( ptr1 + 1 ) != '\0' )
            {
               *ptr1 = '\0';
               ++found;
               break;
            }

         if ( ! found )
         {
            syslog( LOG_ERR, "server specification missing port: %s", *ptr );
            return 1;
         }
      }
   }

   return 0;
}

void read_config()
{
   int fd;
   char *ptr, *host, ***sptr, **servers[] = { &websocket1, &scgi1, &websocket2, &scgi2, &websocket3, &scgi3,
                                              &websocket4, &scgi4, &websocket5, &scgi5, NULL };

   if ( config_file == NULL )
   {
      syslog( LOG_ERR, "-f option missing" );
      exit( 1 );
   }

   if (( fd = open( config_file, O_RDONLY )) < 0 )
   {
      syslog( LOG_ERR, "could not open: %s", config_file );
      exit( 1 );
   }

   if (( ptr = get_line( fd, 1 )) == NULL )
   {
      syslog( LOG_ERR, "could not read hostname from %s", config_file );
      exit( 1 );
   }

   host = set_hostname( ptr );

   if (( ptr = get_line( fd, 1 )) == NULL )
   {
      syslog( LOG_ERR, "could not read TLS keyfile from %s", config_file );
      exit( 1 );
   }

   if (( keyfile = str_dup( ptr, -1 )) == NULL )
      exit( 1 );

   if (( ptr = get_line( fd, 1 )) == NULL )
   {
      syslog( LOG_ERR, "could not read TLS key password from %s", config_file );
      exit( 1 );
   }

   if (( password = str_dup( ptr, -1 )) == NULL )
      exit( 1 );

   if (( ptr = get_line( fd, 1 )) == NULL )
   {
      syslog( LOG_ERR, "could not read TLS chainfile path from %s", config_file );
      exit( 1 );
   }

   chainfile = str_dup( ptr, -1 );

   for( sptr = servers; *sptr != NULL; ++sptr )
   {
      if (( ptr = get_line( fd, 1 )) == NULL )
         goto END;

      **sptr = str_dup( ptr, -1 );
   }

END:
   get_line( -1, 1 );
   close( fd );

   if ( verify_servers())
      exit( 1 );

   make_names( host );
   free( host );
}

void change_directory()
{
   if ( chdir( root ) < 0 )
   {
      syslog( LOG_ERR, "chdir(): %m" );
      exit( 1 );
   }
}

void open_logfile()
{
   char buffer[ MAXPATHLEN ];
   time_t t;
   struct tm *lt;

   sighup = 0;

   if ( logline == NULL && ( logline = make_string()) == NULL )
      return;

   if ( logfile != NULL )
      fclose( logfile );

   if (( logfile = fopen( log_filename, "a" )) == NULL )
   {
      syslog( LOG_ERR, "fopen( %s ): %m", log_filename );
      return;
   }

   t = time( NULL );
   lt = gmtime( &t );
   strftime( buffer, sizeof( buffer ),
             "#Version: 1.0\n"
             "#Date: %Y-%m-%d %H:%M:%S\n"
             "#Fields: date time c-ip cs(User-Agent) cs(Origin) cs(Referer) cs-method cs-uri\n", lt );

   if ( fputs( buffer, logfile ) == EOF )
      syslog( LOG_ERR, "fputs( logfile ): %m" );
}

void set_signals()
{
   int *iptr;
   int sigs[] = {
      SIGPIPE, SIGHUP, SIGQUIT, SIGUSR1, SIGUSR2, SIGALRM, SIGINT, SIGTSTP, -1
   };

   signal( SIGTERM, sigterm_handler );

   for( iptr = sigs; *iptr > 0; ++iptr )
      signal( *iptr, SIG_IGN );
}

int main( int argc, char **argv )
{
   openlog( "prospero", LOG_PID, LOG_DAEMON );
   set_signals();
   set_options( argc, argv );

   read_config();
   init_tls();

   init_content_types();
   change_directory();

   start_listening( &sec_fd, "443" );
   start_listening( &insec_fd, "80" );

   if ( logging )
      open_logfile();

   change_identity();
   process_clients();

   return 0;
}
