/*
 * Avalon Copyright (c) 2020, 2022, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <time.h>
#include <syslog.h>
#include <stdarg.h>

#include "message.h"

/*
 * MAX_CHAT is the maximum size of a client message that we accept.

 * MAX_NAME is the maximum length of user names.  Some UTF-8 sequences are 4
 * bytes long, so a MAX_NAME of 64, allows UTF-8 names from 16 to 64
 * characters, depending on a user's character set.
 */

#define MAX_CHAT 32766
#define MAX_NAME 64
#define TIMEOUT 1800

struct conn
{
   void *conn;
   unsigned char name[ MAX_NAME + 1 ];
   int namelen, authorized, closing, handshook;
   time_t last_message;

} conns[ MAX_CLIENTS ], *conn_ptr;

int num_conns = 0;
unsigned char *password = ( unsigned char *)PASSWORD;

void ms_init_func()
{
   ms_set_name( "avalon" );
   tzset();
   bzero( conns, MAX_CLIENTS * sizeof( struct conn ));
   conn_ptr = conns;
}

void ms_exit_func()
{
}

/*
 * We set the closing flag on connections on which we have called
 * ms_close_conn().  If, on an invocation of timeout_conns(), we find
 * connections still in the closing state, we forcibly close them.  This
 * prevents idle connections from hanging indefinitely.
 */

void timeout_conns()
{
   int i, total;
   time_t t;
   unsigned char buffer[ 128 ];

   total = snprintf(( char * )buffer, sizeof( buffer ), "- You have fallen asleep and floated away from the Isle of Avalon.\n" );
   t = time( NULL );

   for( i = 0; i < MAX_CLIENTS; ++i )
   {
      if ( conns[ i ].conn == NULL )
         continue;

      if ( conns[ i ].closing )
         ms_close_conn( conns[ i ].conn, 1 );
      else if (( t - conns[ i ].last_message ) > TIMEOUT )
      {
         if ( ! conns[ i ].handshook )
            ms_close_conn( conns[ i ].conn, 1 );
         else
         {
            ++conns[ i ].closing;
            ms_queue_message( &conns[ i ].conn, 1, total, buffer );
            ms_close_conn( conns[ i ].conn, 0 );
         }
      }
   }
}  

/*
 * When complain_and_die() is called from open_callback(),
 * complain_and_die() must return 0 so that the library will not try to
 * close a connection that we have closed.  That will crash the server.
 * When we call ms_close_conn() with a second argument of zero, the library
 * closes the connection immediately if the output queue is empty.  The
 * queue will be empty if the call to ms_queue_message() fails.

 * All other locations that call complain_and_die() return to
 * read_callback().  They all set result to 1 to indicate that we have
 * called ms_close_conn().  This prevents the library from freeing the
 * buffer of data passed to read_callback() twice.  Again, this will only
 * happen if ms_queue_message() fails.

 * if ms_queue_message() succeeds, then the connection will be closed when
 * the output queue has been send to the client.
 */

int complain_and_die( void *conn, int result, unsigned int len, unsigned char *buffer )
{
   struct conn *ptr;

   if (( ptr = ms_get_data( conn )) == NULL )
      ms_close_conn( conn, 1 );
   else
   {
      ++ptr->closing;
      ms_queue_message( &conn, 1, len, buffer );
      ms_close_conn( conn, 0 );
   }

   return result;
}

int ms_accept_callback( void *conn )
{
   /*
    * Because we check num_conns against MAX_CLIENTS, we know that there is
    * at least one slot open.  The goto will not create an infinite loop.
    */

AGAIN:
   while( conn_ptr <= &conns[ MAX_CLIENTS - 1 ] && conn_ptr->conn != NULL )
      ++conn_ptr;

   if ( conn_ptr > &conns[ MAX_CLIENTS - 1 ] )
   {
      conn_ptr = conns;
      goto AGAIN;
   }

   conn_ptr->last_message = time( NULL );
   conn_ptr->conn = conn;
   ms_set_data( conn, conn_ptr );

   return 0;
}

int ms_open_callback( void *conn )
{
   unsigned int total;
   unsigned char buffer[ 256 ];
   char timebuf[ 128 ];
   struct tm *lt;
   struct conn *ccb;

   if (( ccb = ms_get_data( conn )) == NULL )
      return 1;

   ++ccb->handshook;
   
   if ( num_conns == MAX_CLIENTS )
   {
      total = snprintf(( char *)buffer, sizeof( buffer ),
                        "- The Isle of Avalon is full.\n"
                        "- Only %d persons may walk the island at one time.\n", MAX_CLIENTS );

      if ( complain_and_die( conn, 0, total, buffer ))
      {
         bzero( ccb, sizeof( struct conn ));
         return 1;
      }
   }

   /*
    * Send the banner and time to the client.
    */

   lt = localtime( &ccb->last_message );
   strftime( timebuf, sizeof( timebuf ), "%A, %d %B %Y, %I:%M:%S %p (%Z)\n", lt );

   total = snprintf(( char *)buffer, sizeof( buffer ),
                     "- Avalon Server " VERSION "\n"
                     "- %s", timebuf );

   if ( ++num_conns == 1 )
      ms_set_periodic( timeout_conns, TIMEOUT );

   if ( ms_queue_message( &conn, 1, total, buffer ))
   {
      bzero( ccb, sizeof( struct conn ));
      return 1;
   }

   return 0;
}

void ms_close_callback( void *conn )
{
   int i, n;
   unsigned int total;
   unsigned char buffer[ MAX_NAME + 128 ];
   struct conn *ptr = NULL;
   void *clients[ MAX_CLIENTS ];

   if (( ptr = ms_get_data( conn )) == NULL )
      return;

   /*
    * If the client is active, send a departure message to other active
    * clients.
    */

   if ( ptr->namelen )
   {
      total = snprintf(( char *)buffer, sizeof( buffer ), "- %s has departed the Isle of Avalon.\n", ptr->name );

      for( i = 0, n = 0; i < MAX_CLIENTS; ++i )
      {
         if ( !conns[ i ].namelen || ptr == &conns[ i ] )
            continue;

         clients[ n++ ] = conns[ i ].conn;
      }

      /*
       * We ignore the return value of ms_queue_message().  If we're low on
       * memory, some connections won't receive the message.  The connection
       * is already in the process of closing.  If we called ms_close_conn(),
       * we would end up in an infinite loop.
       */

      ms_queue_message( clients, n, total, buffer );
   }

   bzero( ptr, sizeof( struct conn ));

   if ( ! --num_conns )
      ms_set_periodic( NULL, 0 );
}

int queue_time( void *conn )
{
   struct tm *lt;
   time_t t;
   unsigned int total;
   unsigned char buffer[ 64 ];

   t = time( NULL );
   lt = localtime( &t );
   total = strftime(( char *)buffer, sizeof( buffer ), "- %I:%M:%S %p (%Z)\n", lt );

   if ( ms_queue_message( conn, 1, total, buffer ))
   {
      struct conn *ptr;

      if (( ptr = ms_get_data( conn )) == NULL )
         ms_close_conn( conn, 1 );
      else
      {
         ++ptr->closing;
         ms_close_conn( conn, 0 );
      }

      return 1;
   }

   return 0;
}

int queue_occupants( struct conn *conn )
{
   int i;
   unsigned int total;
   unsigned char buffer[ MAX_NAME + 4 ];

   /*
    * It is more convenient to fragment the message than it is to assemble
    * the message.  Because only one frame is delivered to clients per write
    * event, a busy system may take some time to deliver the list.
    */

   for( i = 0; i < MAX_CLIENTS; ++i )
   {
      if ( ! conns[ i ].namelen )
         continue;

      total = snprintf(( char *)buffer, sizeof( buffer ), "- %s\n", conns[ i ].name );

      if ( ms_queue_frame( &conn->conn, 1, 0, total, buffer ))
      {
         struct conn *ptr;

         if (( ptr = ms_get_data( conn->conn )) == NULL )
            ms_close_conn( conn->conn, 1 );
         else
         {
            ++ptr->closing;
            ms_close_conn( conn->conn, 0 );
         }

         return 1;
      }
   }

   if ( ms_queue_frame( &conn->conn, 1, 1, 0, buffer ))
   {
      struct conn *ptr;

      if (( ptr = ms_get_data( conn->conn )) == NULL )
         ms_close_conn( conn->conn, 1 );
      else
      {
         ++ptr->closing;
         ms_close_conn( conn->conn, 0 );
      }

      return 1;
   }

   return 0;
}

int authorize( struct conn *conn, unsigned int len, unsigned char *data )
{
   unsigned char buffer[ 128 ], *ptr1, *ptr2;

   ptr1 = password;
   ptr2 = data;

   while( len && *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      --len;
      ++ptr1;
      ++ptr2;
   }

   if ( len || *ptr1 )
   {
      len = snprintf(( char *)buffer, sizeof( buffer ),
                       "- You may not set foot on the isle of avalon.\n"
                       "- Begone.\n" );
      return complain_and_die( conn->conn, 1, len, buffer );
   }

   len = snprintf(( char *)buffer, sizeof( buffer ),
                    "- You may set foot on the Isle of Avalon.\n"
                    "- Enter your user name: " );

   if (( ms_queue_message( &conn->conn, 1, len, buffer )))
   {
      struct conn *ptr;

      if (( ptr = ms_get_data( conn->conn )) == NULL )
         ms_close_conn( conn->conn, 1 );
      else
      {
         ++ptr->closing;
         ms_close_conn( conn->conn, 0 );
      }

      return 1;
   }

   ++conn->authorized;
   return 0;
}

int read_name( struct conn *conn, unsigned int len, unsigned char *data )
{
   unsigned int total, i;
   unsigned char buffer[ MAX_NAME + 128 ], *ptr;

   /*
    * Trim the handle.
    */

   while( len && data[ len - 1 ] <= 32 )
      --len;

   /*
    * Reject bad names.

    * It is more convenient to drop the connection than it is to send the
    * user another name prompt.  The client only has to know that the first
    * outgoing message is the name.  If we give the user multiple
    * opportunities to name itself, we have to send a message to the client
    * to tell it when a name has been accepted.
    */

   if ( ! len || len > MAX_NAME )
   {
      total = snprintf(( char *)buffer, sizeof( buffer ), "- %s.\n",
                        ( len ? "That name is too long" : "A name is required" ));
      return complain_and_die( conn->conn, 1, total, buffer );
   }

   if ( *data == '-' )
   {
      total = snprintf(( char *)buffer, sizeof( buffer ), "- Names cannot start with hyphens.\n" );
      return complain_and_die( conn->conn, 1, total, buffer );
   }

   ptr = data;

   for( i = 0; i < len; --i )
      if ( *ptr++ < 32 )
      {
         total = snprintf(( char *)buffer, sizeof( buffer ), "- Names cannot contain control codes.\n" );
         return complain_and_die( conn->conn, 1, total, buffer );
      }

   conn->namelen = len;
   bcopy( data, conn->name, conn->namelen );

   total = snprintf(( char *)buffer, sizeof( buffer ),
                     "- Welcome to the Isle of Avalon %s.\n"
                     "- ? shows who is here.\n"
                     "- . shows the time.\n", conn->name );

   if ( ms_queue_message( &conn->conn, 1, total, buffer ))
   {
      struct conn *ptr;

      if (( ptr = ms_get_data( conn->conn )) == NULL )
         ms_close_conn( conn->conn, 1 );
      else
      {
         ++ptr->closing;
         ms_close_conn( conn->conn, 0 );
      }

      return 1;
   }

   return 0;
}

/*
 * Returns a pointer to the beginning of a newline-terminated line in the message.
 * Puts the length of the line into line_len.  Puts the start of the next line in
 * *input.
 */

unsigned char *get_line( unsigned char **input, unsigned int *len, unsigned int *line_len )
{
   unsigned char *ptr, *start;

   ptr = start = *input;
   *line_len = 0;

   while( *len && *ptr != '\n' )
   {
      --*len;
      ++ptr;
      ++*line_len;
   }

   if ( ! *len && ! *line_len )
      return NULL;

   if ( *len )
   {
      --*len;
      *input = ptr + 1;
   }

   return start;
}

/*
 * We discard lines that put us over the outgoing message-length limit of
 * MAX_CHAT * 2.
 */

unsigned int compose_message( struct conn *conn, unsigned char *buffer, unsigned int size,
                              unsigned int len, unsigned char *data )
{
   unsigned char *ptr1, *ptr2, *ptr;
   unsigned int total, line_len;

   total = len;

   /*
    * Trim message. Reject blank message.
    */

   while( len && data[ len - 1 ] <= 32 )
      --len;

   if ( ! len )
      return 0;

   len   = total;
   total = 0;
   ptr1  = buffer;

   while(( ptr = get_line( &data, &len, &line_len )) != NULL )
   {
      if (( total + line_len + conn->namelen + 3 ) > size )
         return total;

      total += line_len + conn->namelen + 3;

      ptr2 = conn->name;

      while( *ptr2 )
         *ptr1++ = *ptr2++;

      *ptr1++ = ':';
      *ptr1++ = ' ';

      ptr2 = ptr;

      while( line_len-- )
      {
         if ( *ptr2 == '\t' )
            *ptr2 = ' ';
         else if ( *ptr2 < 32 )
         {
            total = snprintf(( char *)buffer, size, "- Chat data cannot contain control codes.\n" );
            return complain_and_die( conn->conn, 1, total, buffer );
         }

         *ptr1++ = *ptr2++;
      }

      *ptr1++ = '\n';
   }

   return total;
}

int process_command( struct conn *conn, char command )
{
   switch( command )
   {
      case '?':
         return queue_occupants( conn );

      case '.':
         return queue_time( conn );
   }

   return 0;
}

int ms_read_callback( void *conn, unsigned int len, unsigned char *data )
{
   int i, n, new = 0;
   unsigned int total;
   struct conn *ptr = NULL;
   unsigned char buffer[ MAX_CHAT * 2 ];
   void *clients[ MAX_CLIENTS ];

   if ( ! len || len >= MAX_CHAT )
      return 0;

   if (( ptr = ms_get_data( conn )) == NULL )
   {
      ++ptr->closing;
      ms_close_conn( conn, 0 );
      return 1;
   }

if ( ! ptr->authorized )
      return authorize( ptr, len, data );

   ptr->last_message = time( NULL );

   /*
    * If the client is active...
    */

   if ( ptr->namelen )
   {
      if ( len == 1 )
         return process_command( ptr, *data );

      if (( total = compose_message( ptr, buffer, sizeof( buffer ), len, data )) == 1 )
         return total;
   }

   /*
    * If the client is inactive...
    */

   else
   {
      if ( read_name( ptr, len, data ))
         return 1;

      ++new;
      total = snprintf(( char *) buffer, sizeof( buffer ),
                         "- %s has alighted on the Isle of Avalon.\n", ptr->name );
   }

   /*
    * Add the active connections to a list.  Don't send the arrival message
    * to the new connection.  Skip idle connections with more than a
    * hundred outgoing frames queued.
    */

   for( i = 0, n = 0; i < MAX_CLIENTS; ++i )
   {
      if ( ! conns[ i ].namelen || ( new && conns[ i ].conn == conn ) || ms_get_qlen( conns[ i ].conn ) > 100 )
         continue;

      clients[ n++ ] = conns[ i ].conn;
   }

   ms_queue_message( clients, n, total, buffer );
   return 0;
}

void ms_write_callback( void *conn )
{
}
