/*
 * LibYarb Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#define YARB_RED 0
#define YARB_BLACK 1

#include <stdlib.h>
#include <syslog.h>
#include <stdarg.h>

union yarb_keyval
{
   void *ptr;
   float num;
};

struct yarb
{
   union yarb_keyval key, value;
   struct yarb *left, *right, *parent;
   unsigned char color;
};

struct yarb_list
{
   union yarb_keyval key, value;
};

struct yarb_stack
{
   int free, used;
   struct yarb **top, **values;
};

struct yarb_stack *yarb_make_stack();
struct yarb *yarb_stack_push( struct yarb_stack *, struct yarb * );

struct yarb *yarb_lookup( const struct yarb *, int ( * )( const union yarb_keyval, const union yarb_keyval ),
                          const union yarb_keyval );

struct yarb *yarb_insert( struct yarb *, int ( * )( const union yarb_keyval, const union yarb_keyval ),
                          void ( * )( union yarb_keyval ),
                          const union yarb_keyval, const union yarb_keyval );

struct yarb *yarb_delete( struct yarb *, int ( * )( const union yarb_keyval, const union yarb_keyval ),
                          void ( * )( union yarb_keyval, union yarb_keyval ),
                          const union yarb_keyval );

struct yarb *yarb_first( const struct yarb * );
struct yarb *yarb_last( const struct yarb * );

struct yarb *yarb_next( const struct yarb * );
struct yarb *yarb_previous( const struct yarb * );

struct yarb *yarb_traverse( const struct yarb *, void ( * )( struct yarb *, void * ), void *, int );
int yarb_destroy( struct yarb *, void ( * )( union yarb_keyval, union yarb_keyval ));

#define YARB_STACK_PUSH( _s_, _o_ ) ( ! ( _s_ )->free ? yarb_stack_push( _s_, _o_ ) : \
   ( ! ( _s_ )->used ? ( --( _s_ )->free, ++( _s_ )->used, *( _s_ )->top = _o_ ) : \
     ( ++( _s_ )->top, --( _s_ )->free, ++( _s_ )->used, *( _s_ )->top = _o_ )))

#define YARB_STACK_POP( _s_ ) ( !( _s_ )->used ? NULL : ( ++( _s_ )->free, --( _s_ )->used, \
   ((( _s_ )->top == ( _s_ )->values ) ? *( _s_ )->top : ( --( _s_ )->top, *(( _s_ )->top + 1 )))))

#define YARB_STACK_CLEAR( _s_ ) ( _s_ )->free += ( _s_ )->used; ( _s_)->used = 0; ( _s_ )->top = ( _s_ )->values
#define YARB_STACK_FREE( _s_ ) free(( _s_ )->values ); free(( _s_ ))

struct yarb *yarb_lookup( const struct yarb *tree,
                          int ( *comparator )( const union yarb_keyval, const union yarb_keyval ),
                          const union yarb_keyval key )
{
   struct yarb *result = NULL;
   int order;

   for( ; tree != NULL; )

      if (( order = comparator( key, tree->key )) < 0 )
         tree = tree->left;
      else if ( ! order )
      {
         result = ( struct yarb *)tree;
         tree = NULL;
      }
      else
         tree = tree->right;

   return result;
}

struct yarb *yarb_rotate_left( struct yarb *node )
{
   struct yarb *gp, *t;

   gp = node->parent;
   t = node->right;

   if ( gp != NULL )
   {
      if ( gp->right == node )
         gp->right = t;
      else
         gp->left = t;
   }

   t->parent = gp;

   if (( node->right = t->left ) != NULL )
      node->right->parent = node;

   t->left = node;
   node->parent = t;

   return ( gp == NULL ? t : NULL );
}

struct yarb *yarb_rotate_right( struct yarb *node )
{
   struct yarb *gp, *t;

   gp = node->parent;
   t = node->left;

   if ( gp != NULL )
   {
      if ( gp->right == node )
         gp->right = t;
      else
         gp->left = t;
   }

   t->parent = gp;

   if (( node->left = t->right ) != NULL )
      node->left->parent = node;

   t->right = node;
   node->parent = t;

   return ( gp == NULL ? t : NULL );
}

void yarb_rotate_recolor( struct yarb *node, struct yarb **tree )
{
   struct yarb *root, *grandparent, *parent;

   /*
    * Grab pointers to these nodes before their relationships are changed
    * by rotations.
    */

   parent = node->parent;
   grandparent = node->parent->parent;
   grandparent->color = YARB_RED;

   if ( parent->left == node )
   {
      if ( grandparent->left == parent )
      {
         if (( root = yarb_rotate_right( grandparent )) != NULL )
            *tree = root;

         parent->color = YARB_BLACK;
      }
      else
      {
         if (( root = yarb_rotate_right( parent )) != NULL )
            *tree = root;

         if (( root = yarb_rotate_left( grandparent )) != NULL )
            *tree = root;

         node->color = YARB_BLACK;
      }
   }
   else
   {
      if ( grandparent->right == parent )
      {
         if (( root = yarb_rotate_left( grandparent )) != NULL )
            *tree = root;

         parent->color = YARB_BLACK;
      }
      else
      {
         if (( root = yarb_rotate_left( parent )) != NULL )
            *tree = root;

         if (( root = yarb_rotate_right( grandparent )) != NULL )
            *tree = root;

         node->color = YARB_BLACK;
      }
   }
}

void yarb_balance_insertion( struct yarb *node, struct yarb **tree )
{
   struct yarb *uncle;

   for( ; ; )
   {
      if ( node->parent == NULL )
      {
         node->color = YARB_BLACK;
         break;
      }

      if ( node->parent->color == YARB_BLACK )
         break;

      if ( node->parent->parent->left == node->parent )
         uncle = node->parent->parent->right;
      else
         uncle = node->parent->parent->left;

      if ( uncle == NULL || uncle->color == YARB_BLACK )
      {
         yarb_rotate_recolor( node, tree );
         break;
      }

      node->parent->color = uncle->color = YARB_BLACK;
      node->parent->parent->color = YARB_RED;
      node = node->parent->parent;
   }
}

struct yarb *yarb_insert( struct yarb *tree,
                          int ( *comparator )( const union yarb_keyval, const union yarb_keyval ),
                          void ( *func )( const union yarb_keyval ),
                          const union yarb_keyval key, const union yarb_keyval value )
{
   struct yarb *parent = NULL, **ptr, *node = NULL;
   int order;

   for( node = tree; node != NULL; )

      if (( order = comparator( key, node->key )) < 0 )
      {
         parent = node;
         ptr = &node->left;
         node = node->left;
      }
      else if ( ! order )
         break;
      else
      {
         parent = node;
         ptr = &node->right;
         node = node->right;
      }

   if ( node != NULL )
   {
      if ( func != NULL )
         func( node->value );
   }
   else
   {
      if (( node = malloc( sizeof( struct yarb ))) == NULL )
         return NULL;

      node->key = key;
      node->parent = parent;
      node->left = node->right = NULL;
      node->color = YARB_RED;

      if ( tree != NULL )
         *ptr = node;
      else
         tree = node;

      yarb_balance_insertion( node, &tree );
   }

   node->value = value;
   return tree;
}

struct yarb *yarb_red_nephew_1( int left_child, struct yarb *sibling, struct yarb **tree )
{
   struct yarb *root;

   if ( left_child && sibling->left->color == YARB_RED && sibling->right->color == YARB_BLACK )
   {
      sibling->left->color = YARB_BLACK;

      if (( root = yarb_rotate_right( sibling )) != NULL )
         *tree = root;

      sibling = sibling->left;
   }
   else if ( ! left_child && sibling->left->color == YARB_BLACK && sibling->right->color == YARB_RED )
   {
      sibling->right->color = YARB_BLACK;

      if (( root = yarb_rotate_left( sibling )) != NULL )
         *tree = root;

      sibling = sibling->right;
   }

   return sibling;
}

void yarb_red_nephew_2( int left_child, struct yarb *parent, struct yarb *sibling, struct yarb **tree )
{
   struct yarb *root;
   unsigned char color;

   if ( left_child && parent->left != NULL && sibling->right->color == YARB_RED )
   {
      color = parent->color;
      parent->color = YARB_BLACK;
      sibling->color = color;

      if (( root = yarb_rotate_right( parent )) != NULL )
         *tree = root;
   }
   else if ( ! left_child && parent->right != NULL & sibling->left->color == YARB_RED )
   {
      color = parent->color;
      parent->color = YARB_BLACK;
      sibling->color = color;

      if (( root = yarb_rotate_left( parent )) != NULL )
         *tree = root;
   }
}

void yarb_balance_deletion( int left_child, struct yarb *parent, struct yarb **tree )
{
   struct yarb *root, *sibling;

   for( ; ; )
   {
      if ( parent == NULL )
         return;

      if (( sibling = ( left_child ? parent->right : parent->left )) == NULL )
         return;

      if ( sibling->color == YARB_RED )
      {
         parent->color = YARB_RED;
         sibling->color = YARB_BLACK;

         if ( left_child )
         {
            if (( root = yarb_rotate_left( parent )) != NULL )
               *tree = root;

            sibling = parent->right;
         }
         else
         {
            if (( root = yarb_rotate_right( parent )) != NULL )
               *tree = root;

            sibling = parent->left;
         }
      }

      if ( sibling == NULL || sibling->left == NULL || sibling->right == NULL )
         return;

      if ( parent->color         == YARB_BLACK && sibling->color        == YARB_BLACK &&
           sibling->left->color  == YARB_BLACK && sibling->right->color == YARB_BLACK )
      {
         sibling->color = YARB_RED;
         parent = parent->parent;

         if ( parent != NULL )
            left_child = ( parent->left == parent ? 1 : 0 );

         continue;
      }

      if ( parent->color        == YARB_RED   && sibling->color        == YARB_BLACK &&
           sibling->left->color == YARB_BLACK && sibling->right->color == YARB_BLACK )
      {
         sibling->color = YARB_RED;
         return;
      }

      if ( sibling->color == YARB_BLACK )
         sibling = yarb_red_nephew_1( left_child, sibling, tree );

      if ( sibling       != NULL && sibling->color == YARB_BLACK &&
           sibling->left != NULL && sibling->right != NULL )
         yarb_red_nephew_2( left_child, parent, sibling, tree );

      return;
   }
}

struct yarb *yarb_delete( struct yarb *tree,
                          int ( *comparator )( const union yarb_keyval, const union yarb_keyval ),
                          void ( *func )( union yarb_keyval, union yarb_keyval ),
                          const union yarb_keyval key )
{
   struct yarb *node, *result = tree;

   if (( node = yarb_lookup( tree, comparator, key )) == NULL )
      return result;

   if ( func != NULL )
      func( node->key, node->value );

   if ( node->left == NULL && node->right == NULL )
   {
      int left_child;

      if ( node->parent == NULL )
         result = NULL;
      else if ( node->parent->left == node )
      {
         left_child = 1;
         node->parent->left = NULL;
      }
      else
      {
         left_child = 0;
         node->parent->right = NULL;
      }

      if ( result != NULL && node->color == YARB_BLACK )
      {
         yarb_balance_deletion( left_child, node->parent, &tree );
         result = tree;
      }

      free( node );
   }
   else if ( node->left != NULL && node->right != NULL )
   {
      result = yarb_next( node );

      node->key = result->key;
      node->value = result->value;

      yarb_delete( node->right, comparator, func, node->key );
      result = tree;
   }
   else
   {
      if ( node->left == NULL )
         result = node->right;
      else
         result = node->left;

      node->key = result->key;
      node->value = result->value;

      node->left = result->left;
      node->right = result->right;

      if ( node->left != NULL )
         node->left->parent = node;

      if ( node->right != NULL )
         node->right->parent = node;

      free( result );
      result = tree;
   }

   return result;
}

struct yarb *yarb_previous( const struct yarb *tree )
{
   const struct yarb *node;

   if ( tree == NULL )
      return NULL;

   for( node = tree->left; node != NULL; node = node->right )
      if ( node->right == NULL )
         return ( struct yarb *)node;

   node = tree;

   if ( node->parent == NULL )
      return NULL;

   if ( node->parent->right == node )
      return node->parent;

   for( node = node->parent; node != NULL; node = node->parent )
      if ( node->parent != NULL && node->parent->right == node )
         return node->parent;

   return NULL;
}

struct yarb *yarb_next( const struct yarb *tree )
{
   const struct yarb *node;

   if ( tree == NULL )
      return NULL;

   for( node = tree->right; node != NULL; node = node->left )
      if ( node->left == NULL )
         return ( struct yarb *)node;

   node = tree;

   if ( node->parent == NULL )
      return NULL;

   if ( node->parent->left == node )
      return node->parent;

   for( node = node->parent; node != NULL; node = node->parent )
      if ( node->parent != NULL && node->parent->left == node )
         return node->parent;

   return NULL;
}

struct yarb *yarb_first( const struct yarb *tree )
{
   while( tree != NULL && tree->left != NULL )
      tree = tree->left;

   return ( struct yarb *)tree;
}

struct yarb *yarb_last( const struct yarb *tree )
{
   while( tree != NULL && tree->right != NULL )
      tree = tree->right;

   return ( struct yarb *)tree;
}

struct yarb *yarb_stack_push( struct yarb_stack *a, struct yarb *o )
{
   if ( a->free == 0 )
   {
      a->values = realloc( a->values, sizeof( void * ) * ( a->used + 64 ));

      if ( a->values == NULL )
      {
         syslog( LOG_ERR, "realloc: %m.\n" );
         return NULL;
      }

      a->free = 64;
      a->top = &a->values[ a->used - 1 ];
   }

   if ( a->used )
      ++a->top;

   *a->top = o;
   --a->free;
   ++a->used;

   return o;
}

struct yarb_stack *yarb_make_stack()
{
   struct yarb_stack *a;

   if (( a = malloc( sizeof( struct yarb_stack ))) == NULL )
   {
      syslog( LOG_ERR, "malloc(): %m" );
      return NULL;
   }

   if (( a->values = malloc( sizeof( void * ) * 64 )) == NULL )
   {
      syslog( LOG_ERR, "malloc(): %m" );
      free( a );
      return NULL;
   }

   a->free = 64;
   a->used = 0;
   a->top = a->values;

   return a;
}

struct yarb *yarb_traverse( const struct yarb *tree, void ( *func )( struct yarb *, void * ),
                            void *data, int order )
{
   struct yarb *node;
   struct yarb_stack *stack;

   if ( tree == NULL )
      return NULL;

   if (( stack = yarb_make_stack()) == NULL )
      return NULL;

   for( node = ( struct yarb *)tree; ; )
   {
      for( ; node != NULL; node = ( order < 0 ? node->right : node->left ))
         if ( YARB_STACK_PUSH( stack, node ) == NULL )
            return NULL;

      if (( node = YARB_STACK_POP( stack )) == NULL )
         break;

      func( node, data );
      node = ( order < 0 ? node->left : node->right );
   }

   YARB_STACK_FREE( stack );
   return ( struct yarb *)tree;
}

int yarb_destroy( struct yarb *tree, void ( *func )( union yarb_keyval, union yarb_keyval ))
{
   struct yarb *node;
   struct yarb_stack *stack;

   if ( tree == NULL )
      return 0;

   if (( stack = yarb_make_stack()) == NULL )
      return 1;

   for( ; ; )
   {
      for( node = tree; node != NULL; node = node->left )
         if ( YARB_STACK_PUSH( stack, node ) == NULL )
            return 1;

      if (( node = YARB_STACK_POP( stack )) == NULL )
         break;

      if ( func != NULL )
         func( node->key, node->value );

      tree = node->right;
      free( node );
   }

   YARB_STACK_FREE( stack );
   return 0;
}
