/*
 * LibIndex Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>

#include "yarb.h"

#define MAX_NODES 1024

int number_comparator( const union yarb_keyval first, const union yarb_keyval second )
{
   return first.num - second.num;
}

void num_parents( struct yarb *node, void *data )
{
   printf( "%f -> %f\n", node->key.num, ( node->parent == NULL ? -1 : node->parent->key.num ));

   if ( data != NULL )
      YARB_STACK_PUSH(( struct yarb_stack *)data, node );
}

void progress( struct yarb *node, void *data )
{
   printf( "%f ", node->key.num );
}

void numerical_keys()
{
   struct yarb *tree = NULL, *node, **ptr;
   struct yarb_stack *stack;
   struct yarb_list index;
   int n;

   /*
    * This is a contrived example.  If you had contiguous numerical
    * keys, you would, of course, use an array.
    */

   printf( "insert check (%d keys)...\n", MAX_NODES );

   for( n = 0; n < MAX_NODES; ++n )
   {
      index.key.num = index.value.num = n;
      tree = yarb_insert( tree, number_comparator, NULL, index.key, index.value );
   }

   puts( "yarb_traverse( 1 )..." );
   yarb_traverse( tree, progress, NULL, 1 );

   puts( "\nparent pointer check:" );
   stack = yarb_make_stack();
   yarb_traverse( tree, num_parents, stack, 1 );

   puts( "stack check:" );
   for( n = 0, ptr = stack->values; n < stack->used; ++n, ++ptr )
      printf( "%f\n", ( *ptr )->key.num );

   YARB_STACK_FREE( stack );

   printf( "yarb_lookup() keys 0, %d, and %d...\n", MAX_NODES - 1, MAX_NODES / 2 );

   index.key.num = 0;
   if (( node = yarb_lookup( tree, number_comparator, index.key )) != NULL )
      printf( "%f -> %f\n", node->key.num, node->value.num );

   index.key.num = MAX_NODES - 1;
   if (( node = yarb_lookup( tree, number_comparator, index.key )) != NULL )
      printf( "%f -> %f\n", node->key.num, node->value.num );

   index.key.num = MAX_NODES / 2;
   if (( node = yarb_lookup( tree, number_comparator, index.key )) != NULL )
      printf( "%f -> %f\n", node->key.num, node->value.num );

   puts( "successor/predecessor checks:" );

   node = yarb_first( tree );
   printf( "yarb_first(): %f\n", node->key.num );

   while(( node = yarb_next( node )) != NULL )
      printf( "yarb_next(): %f\n", node->key.num );

   node = yarb_last( tree );
   printf( "yarb_last(): %f\n", node->key.num );

   while(( node = yarb_previous( node )) != NULL )
      printf( "yarb_previous(): %f\n", node->key.num );

   printf( "delete check 1 (%d keys)...\n", MAX_NODES );

   for( n = 0; n < MAX_NODES; ++n )
   {
      index.key.num = n;
      tree = yarb_delete( tree, number_comparator, NULL, index.key );
   }

   printf( "Tree is %s\n", ( tree == NULL ? "NULL" : "not-NULL" ));
   printf( "delete check 2 (%d keys)...\n", MAX_NODES );

   for( n = 0; n < MAX_NODES; ++n )
   {
      index.key.num = index.value.num = n;
      tree = yarb_insert( tree, number_comparator, NULL, index.key, index.value );
   }

   for( n = MAX_NODES - 1; n >= 0; --n )
   {
      index.key.num = n / 2;
      printf( "deleting key %f\n", index.key.num );
      tree = yarb_delete( tree, number_comparator, NULL, index.key );
      yarb_traverse( tree, progress, NULL, 1 );
      putchar( '\n' );
   }

   printf( "Tree is %s\n", ( tree == NULL ? "NULL" : "not-NULL" ));
   yarb_traverse( tree, progress, NULL, 1 );
   
   puts( "yarb_destroy()..." );
   yarb_destroy( tree, NULL );
}

int main( int argc, char **argv )
{
   numerical_keys();
   return 0;
}
