/*
 * LibTable Copyright (c) 2017, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>

union table_key
{
   char *str;
   int idx;
};

union table_value
{
   void *ptr;
   int num;
};

struct table_element
{
   union table_key key;
   union table_value value;
   struct table_element *next;
};

#define TABLE_STRING 0
#define TABLE_INTEGER 1

struct table
{
   int size, type;
   struct table_element **data;
};

struct table *table_make( int, int );
void table_destroy( struct table *, void ( * )( union table_key, union table_value ));
struct table *table_resize( struct table *, int );

void table_traverse( struct table *table,
                     void ( *func )( struct table_element *, void * ),
                     void *data );

int table_insert( struct table *, void ( * )( union table_key, union table_value ), union table_key, union table_value );
void table_delete( struct table *, void ( * )( union table_key, union table_value ), union table_key );

struct table_element *table_lookup( struct table *, union table_key );

struct table *table_make( int type, int size )
{
   struct table *t;

   if (( t = malloc( sizeof( struct table ))) == NULL )
      return NULL;

   t->size = size;

   if (( t->data = calloc( size, sizeof( struct table_element * ))) == NULL )
   {
      free( t );
      return NULL;
   }

   t->type = type;
   return t;
}

void table_destroy( struct table *table, void ( *func )( union table_key key, union table_value value ))
{
   struct table_element **ptr, *ptr2, *ptr3;
   int n;

   for( ptr = table->data, n = 0; n < table->size; ++n, ++ptr )
   {
      for( ptr3 = NULL, ptr2 = *ptr; ptr2 != NULL; ptr2 = ptr2->next )
      {
         if ( func != NULL )
            func( ptr2->key, ptr2->value );

         if ( ptr3 != NULL )
            free( ptr3 );

         ptr3 = ptr2;
      }

      if ( ptr3 != NULL )
         free( ptr3 );
   }

   free( table->data );
   free( table );
}

void table_traverse( struct table *table, void ( *func )( struct table_element *, void * ), void *data )
{
   struct table_element **ptr, *ptr2;
   int n;

   if ( func == NULL )
      return;

   for( ptr = table->data, n = 0; n < table->size; ++n, ++ptr )
      for( ptr2 = *ptr; ptr2 != NULL; ptr2 = ptr2->next )
         func( ptr2, data );
}

struct table *table_resize( struct table *table, int size )
{
   struct table *new;
   struct table_element **ptr, *ptr2;
   int n;

   if (( new = table_make( table->type, size )) == NULL )
      return NULL;

   for( ptr = table->data, n = 0; n < table->size; ++n, ++ptr )
   {
      for( ptr2 = *ptr; ptr2 != NULL; ptr2 = ptr2->next )
      {
         if ( table_insert( new, NULL, ptr2->key, ptr2->value ))
         {
            table_destroy( new, NULL );
            return NULL;
         }
      }
   }

   table_destroy( table, NULL );
   return new;
}

int table_insert( struct table *table, void ( *func )( union table_key, union table_value ), union table_key key, union table_value value )
{
   unsigned long idx;
   char *rptr;
   struct table_element *ptr = NULL, *ptr2 = NULL;

   if ( table->type == TABLE_STRING )
   {
      for( idx = 2166136261, rptr = key.str; *rptr; ++rptr )
         idx = ( idx * 16777619 ) ^ *rptr;
   }
   else if ( table->type == TABLE_INTEGER )
   {
      idx = key.idx;
      idx = ( idx + 0x7ed55d16 ) + ( idx << 12 );
      idx = ( idx ^ 0xc761c23c ) ^ ( idx >> 19 );
      idx = ( idx + 0x165667b1 ) + ( idx << 5 );
      idx = ( idx + 0xd3a2646c ) ^ ( idx << 9 );
      idx = ( idx + 0xfd7046c5 ) + ( idx <<3 );
      idx = ( idx ^ 0xb55af09 ) ^ ( idx >> 16 );
   }
   else
      return -1;

   idx %= table->size;

   if ( table->data[ idx ] == NULL )
   {
      if (( table->data[ idx ] = malloc( sizeof( struct table_element ))) == NULL )
         return 1;

      table->data[ idx ]->key = key;
      table->data[ idx ]->value = value;
      table->data[ idx ]->next = NULL;
      return 0;
   }

   ptr2 = table->data[ idx ];

   for( ptr = ptr2; ptr != NULL; ptr = ptr->next )
   {
      if ( table->type == TABLE_INTEGER )
      {
         if ( ptr->key.idx == key.idx )
            break;
      }
      else
      {
         char *ptr3, *ptr4;

         ptr3 = ptr->key.str;
         ptr4 = key.str;

         while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
         {
            ++ptr3;
            ++ptr4;
         }

         if ( ! *ptr3 && ! *ptr4 )
            break;
      }

      ptr2 = ptr;
   }

   if ( ptr != NULL )
   {
      if ( func != NULL )
         func( ptr->key, ptr->value );

      ptr->key = key;
      ptr->value = value;
   }
   else
   {
      if (( ptr2->next = malloc( sizeof( struct table_element ))) == NULL )
         return 1;

      ptr = ptr2->next;
      ptr->key = key;
      ptr->value = value;
      ptr->next = NULL;
   }

   return 0;
}

void table_delete( struct table *table, void ( *func )( union table_key, union table_value ), union table_key key )
{
   unsigned long idx;
   char *rptr;
   struct table_element *ptr, *ptr2;

   if ( table->type == TABLE_STRING )
   {
      for( idx = 2166136261, rptr = key.str; *rptr; ++rptr )
         idx = ( idx * 16777619 ) ^ *rptr;
   }
   else if ( table->type == TABLE_INTEGER )
   {
      idx = key.idx;
      idx = ( idx + 0x7ed55d16 ) + ( idx << 12 );
      idx = ( idx ^ 0xc761c23c ) ^ ( idx >> 19 );
      idx = ( idx + 0x165667b1 ) + ( idx << 5 );
      idx = ( idx + 0xd3a2646c ) ^ ( idx << 9 );
      idx = ( idx + 0xfd7046c5 ) + ( idx <<3 );
      idx = ( idx ^ 0xb55af09 ) ^ ( idx >> 16 );
   }
   else
      return;

   idx %= table->size;

   if ( table->data[ idx ] == NULL )
      return;

   ptr = table->data[ idx ];
   ptr2 = NULL;

   do
   {
      if ( table->type == TABLE_INTEGER )
      {
         if ( ptr->key.idx == key.idx )
            break;
      }
      else 
      {
         char *ptr3, *ptr4;

         ptr3 = ptr->key.str;
         ptr4 = key.str;

         while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
         {
            ++ptr3;
            ++ptr4;
         }

         if ( ! *ptr3 && ! *ptr4 )
            break;
      }

      ptr2 = ptr;
      ptr = ptr->next;
   }
   while( ptr != NULL );

   if ( ptr == NULL )
      return;

   if ( func != NULL )
      func( ptr->key, ptr->value );

   if ( ptr2 == NULL )
      table->data[ idx ] = ptr->next;
   else
      ptr2->next = ptr->next;

   free( ptr );
   return;
}

struct table_element *table_lookup( struct table *table, union table_key key )
{
   unsigned long idx;
   char *rptr;
   struct table_element *ptr;

   if ( table->type == TABLE_STRING )
   {
      for( idx = 2166136261, rptr = key.str; *rptr; ++rptr )
         idx = ( idx * 16777619 ) ^ *rptr;
   }
   else if ( table->type == TABLE_INTEGER )
   {
      idx = key.idx;
      idx = ( idx + 0x7ed55d16 ) + ( idx << 12 );
      idx = ( idx ^ 0xc761c23c ) ^ ( idx >> 19 );
      idx = ( idx + 0x165667b1 ) + ( idx << 5 );
      idx = ( idx + 0xd3a2646c ) ^ ( idx << 9 );
      idx = ( idx + 0xfd7046c5 ) + ( idx <<3 );
      idx = ( idx ^ 0xb55af09 ) ^ ( idx >> 16 );
   }
   else
      return NULL;

   idx %= table->size;

   for( ptr = table->data[ idx ]; ptr != NULL; ptr = ptr->next )
   {
      if ( table->type == TABLE_INTEGER )
      {
         if ( ptr->key.idx == key.idx )
            break;
      }
      else 
      {
         char *ptr3, *ptr4;

         ptr3 = ptr->key.str;
         ptr4 = key.str;

         while( *ptr3 && *ptr4 && *ptr3 == *ptr4 )
         {
            ++ptr3;
            ++ptr4;
         }

         if ( ! *ptr3 && ! *ptr4 )
            break;
      }
   }

   return ptr;
}
