.Dd Thu Feb 08, 2024
.Dt libstringstack 3
.Sh NAME
.Nm libstringstack
.Nd Dynamically resized stacks and strings
.Sh SYNOPSIS
.Nm #include <stringstack.h>
.Nm -I/usr/local/include -L/usr/local/lib -lstringstack
.Sh DESCRIPTION
LibStringStack provides dynamically resized stacks and
character arrays.
.Pp
All library functions are reentrant if concurrent invocations are applied
to different structures.
.Sh USAGE
The library uses structures defined in stringstack.h:
.Ss STACKS
.Bd -literal -offset left
union stack_val
{
   float num;
   void *ptr;
};

struct stack
{
   int free, used;
   union stack_val *top, *values;
};
.Pp
The free and used elements describe the number of free and used elements in
the values list.
.Pp
The values list contains a contiguous array of union stack_val pointers of
<used> length followed by <free> empty slots.
.Pp
The top element points to the union stack_val on the top of the stack.  If
the stack is empty, top points to the first slot in the values list.
.Pp
stack_make() creates stacks with 64 free slots in the values list.  Pushing
more than 64 elements causes the values list to be resized to its current
size plus 64 more slots.  Stacks never shrink.
.Pp
You iterate over a stack like this:
.Bd -literal -offset left
int i;
union stack_val *p;

for( i = 0, p = stack->values; i < stack->used; ++i, ++p )
   do_something_with( p );
.Ed
.Ss STRINGS
.Bd -literal -offset left
struct string
{
   int free, used;
   char *top, *str;
};
.Ed
.Pp
The free and used elements describe the number of free and used characters
in the str array.
.Pp
The str array contains a contiguous array of characters of <used>
length followed by the zero terminator, followed by <free> empty slots.
.Pp
The top element points to zero terminator in the str array.  If the stack
is empty, the terminator will be in the first slot in the str array.
.Pp
string_make() creates strings with 128 free characters in the str array.
Pushing more than 128 characters causes the str array to be resized to its
current size plus 128 more characters.  Strings never shrink.
.Ss STACK_MAKE()
Create a new stack with stack_make().
.Bd -literal -offset left
struct stack *stack_make();
.Ed
.Pp
stack_make() returns a pointer to a new stack initialized for use.  If
memory allocation fails, stack_make() returns NULL.
.Ss STACK_FREE()
Deallocate stacks with stack_free().
.Bd -literal -offset left
void stack_free( struct stack *stack );
.Ed
.Ss STACK_CLEAR()
Empty a stack with stack_clear().
.Bd -literal -offset left
void stack_clear( struct stack *stack );
.Ed
.Ss STACK_PUSH()
Push an element onto the top of a stack with stack_push().
.Bd -literal -offset left
struct stack *stack_push( struct stack *stack, union stack_val item );
.Ed
.Pp
stack_push() returns 0 on success, or 1 if memory allocation fails.
.Pp
The first argument is the stack.
.Pp
The second argument is the item to be pushed onto the stack.  The content
of the union stack_val is copied into the stack by structure assignment.
.Ss STACK_POP()
Pop the top element off a stack with stack_pop().
.Bd -literal -offset left
union stack_val *stack_pop( struct stack *stack );
.Ed
.Pp
stack_pop() returns NULL if the stack is empty.  On success,
stack_pop() returns a pointer to the element that was on top of the stack
before the call to stack_pop().  The element will be overwritten
by the next call to stack_push().  To preserve the element, assign its data
to another union stack_val with structure assignment.
.Bd -literal -offset left
union stack_val temp, *ptr;

if (( ptr = stack_pop( stack )) != NULL )
   temp = *ptr;
.Ed
.Ss STACK_UNSHIFT()
Push an element onto the bottom of the stack with stack_unshift().
.Bd -literal -offset left
int stack_unshift( struct stack *stack, union stack_val item );
.Ed
.Pp
stack_unshift() returns 0 on success, or 1 if memory allocation fails.
.Pp
The first argument is the stack.
.Pp
The second argument is the item to be pushed onto the stack.  The content
of the union stack_val is copied into the stack by structure assignment.
.Ss STACK_SHIFT()
Pop the bottom element off a stack with stack_shift().
.Bd -literal -offset left
int stack_shift( struct stack *stack, union stack_val *item );
.Ed
.Pp
The first argument is the stack.
.Pp
The second argument is a pointer to a union stack_val to hold the result.
.Pp
stack_shift() returns 1 if the stack is empty.  On success, stack_shift()
returns 0.  On success, the union stack_val pointed to by the second
argument contains the content of the element that was on the bottom of the
stack before stack_shift() was invoked.  The result is copied by structure
assignment.
.Ss STRING_MAKE()
Create a new string with string_make().
.Bd -literal -offset left
struct string *string_make( char *source );
.Ed
.Pp
The argument is a source character array to initialize the string.  The
argument can be NULL.
.Ss STRING_FREE()
Deallocate a string with string_free().
.Bd -literal -offset left
void string_free( struct string * );
.Ed
.Ss STRING_CLEAR()
Empty a string with string_clear().
.Bd -literal -offset left
void string_clear( struct string *string );
.Ed
.Ss STRING_APPEND()
Add a single character to the end of a string with string_append().
.Bd -literal -offset left
int string_append( struct string *string, char c );
.Ed
.Pp
string_append() returns 0 on success or 1 if memory allocation fails.
You can append zero to a string.
.Pp
The first argument is the string.
.Pp
The second argument is the character to be appended.
.Ss STRING_PREPEND()
Add a single character to the front of a string with string_prepend().
.Bd -literal -offset left
int string_prepend( struct string *string, char c );
.Ed
.Pp
string_prepend() returns 0 on success or 1 if memory allocation fails.
You can prepend 0 to a string.
.Pp
.Pp
The first argument is the string.
.Pp
The second argument is the character to be prepended.
.Ss STRING_ASSIGN()
Assign a character array to a string with string_assign().
.Bd -literal -offset left
int string_assign( struct string *string, char *array );
.Ed
.Pp
string_assign() returns 0 on success or 1 if memory allocation fails.
The previous content of the string is discarded. You cannot append a string
containing 0.
.Pp
The first argument is the string.
.Pp
The second argument is the character array to be assigned to the string.
.Ss STRING_CONCAT()
Append a character array to a string with string_concat().
.Bd -literal -offset left
int string_concat( struct string *string, char *array );
.Ed
.Pp
string_concat() returns 0 on success or 1 if memory allocation fails.
The character array is appended to the current content of the string.
You cannot concat a string containing 0.
.Pp
The first argument is the string.
.Pp
The second argument is the character array.
.Ss STRING_PRECAT()
Prepend a character array to a string with string_concat().
.Bd -literal -offset left
int string_precat( struct string *string, char *array );
.Ed
.Pp
string_precat() returns 0 on success or 1 if memory allocation fails.
The character array is prepended to the current content of the string.
You cannot prepend a string containing 0.
.Pp
The first argument is the string.
.Pp
The second argument is the character array.
.Ss STRING_MERGE()
Append the content of one string to another string with string_merge().
.Bd -literal -offset left
int string_merge( struct string *a, struct string *b );
.Ed
.Pp
string_merge() returns 0 on success or 1 if memory allocation fails.  On
success, string b's content is appended to the content of string a.  String
b is unchanged.
.Pp
The first argument is the string to be altered.
.Pp
The second argument is the source string.
.Pp
If you do not want either string to be affected, use
.Bd -literal -offset left
struct string *new = string_make( a->str );
string_concat( new, b->str );
.Ed
.Ss STRING_UTF8_LEN()
Count the length of an UTF-8 string with string_UTF8_len().
.Bd -literal -offset left
unsigned int string_UTF8_len( unsigned char * );
.Ed
.Ss STRING_UTF16_LEN()
Count the length of an UTF-16 string with string_UTF16_len().
.Bd -literal -offset left
unsigned int string_UTF16_len( unsigned char * );
.Ed
.Ss STRING_UTF8_TO_VALUE()
Convert a single UTF-8 character into an unsigned int with string_UTF8_to_value().
.Bd -literal -offset left
unsigned int string_UTF8_to_value( unsigned char * );
.Ed
.Ss STRING_UTF8_TO_UTF32BE()
Convert an UTF-8 string to an UTF-32BE string with string_UTF8_to_UTF32BE().
.Bd -literal -offset left
struct string *string_UTF8_to_UTF32BE( struct string * );
.Ed
.Ss STRING_UTF8_TO_UTF16BE()
Convert an UTF-8 string into an UTF-16BE string with string_UTF8_to_UTF16BE().
.Bd -literal -offset left
struct string *string_UTF8_to_UTF16BE( struct string * );
.Ed
.Ss STRING_UTF16_TO_UTF8()
Convert an UTF16 string of either endianness to an UTF-8 string with string_UTF16_to_UTF8().
.Bd -literal -offset left
struct string *string_UTF16_to_UTF8( struct string * );
.Ss STRING_UTF32BE_TO_UTF8()
Convert an UTF-32BE string into an UTF-8 string with string_UTF32BE_to_UTF8().
.Bd -literal -offset left
struct string *string_UTF32BE_to_UTF8( struct string * );
.Ed
.Sh AUTHORS
.An James Bailie Aq bailie9@icloud.com
.br
mammothcheese.ca
