/*
 * Pip Copyright (c) 2010-2018, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * A kqueue-based, even-driven generic SCGI server as a library.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/un.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>

#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include <openssl/evp.h>
#include <openssl/err.h>
#include <openssl/rand.h>

/*
 * Size of the output queue used by kevent().
 */

#define QLEN 16384

/*
 * Maximum size of chunks of data read and written from/to client.  We
 * multiplex by only processing this much data each time we service a
 * client connection.
 */

#define IOBUFSIZE 32768

/*
 * KQueue Input and output queues.  We can add a maximum of 10 events to
 * the input queue for every item in the output queue, while processing
 * the output queue.
 */

#define MAX_EVENT 10
struct kevent *pip_inqueue, *pip_outqueue;

#define MAX_ENV 50
#define MAX_PARAM 100
#define MAX_COOKIE 50

extern char *optarg;

int pip_in = 0, pip_out = 0, pip_fd = -1, pip_logging = 0, pip_testing = 0,
    pip_backlog = 1024, pip_closed = 0, pip_killed = 0, pip_active = 0, pip_qlen = QLEN,
    pip_timer = 0, pip_max_body = 4096, pip_alloc_err = 0,
    pip_max_conn = QLEN;

char *pip_interface = "", *pip_port = "4000", *pip_group = "nobody", *pip_user = "nobody",
     *pip_config_file = NULL, *pip_root_dir = NULL, *pip_listen_unix = NULL,
     *pip_app_name = "server", *pip_pidfile = "/var/run/pip.pid";

void ( *pip_periodic )() = NULL;

struct passwd *pip_passwd;
struct group *pip_grp;

struct pip_buffer {
   off_t total;
   char *data;
   struct pip_buffer *next;
};

struct pip_string
{
   int free, used;
   char *top;
   char *str;
};

union pip_user_data
{
   void *ptr;
   float num;
};

struct pip_ccb {
   union pip_user_data user_data;
   off_t count, offset;
   int sock, qlen, state;
   char *header, *body, *cookie, *query;
   char inbuf[ 8 ], *env[ MAX_ENV + 1 ], *params[ MAX_PARAM + 1 ], *cookies[ MAX_COOKIE + 1 ];
   struct pip_buffer *queue, *last;
};

int pip_stack_inc = 16;

void pip_init_func();
void pip_exit_func();
int pip_request_handler( void * );

void pip_get_data( void * );
void pip_set_name( char * );
void pip_set_periodic( void (*)(), int );

int pip_write_conn( void *, void *, int );
void pip_free_names( char ** );

void pip_process_cookies( struct pip_ccb * );

void *pip_memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
   {
      if ( pip_logging )
         syslog( LOG_WARNING, "malloc(): %m" );
      else
         fprintf( stderr, "%s: malloc(): %s\n", pip_app_name, strerror( errno ));

      return NULL;
   }

   return ptr;
}

char *pip_str_dup( char *str, int len )
{
   char *ptr;

   if ( len < 0 )
      for( len = 0, ptr = str; *ptr; ++ptr )
         ++len;

   if (( ptr = ( char *)pip_memory( len + 1 )) == NULL )
      return NULL;

   bcopy( str, ptr, len );
   ptr[ len ] = '\0';

   return ptr;
}

struct pip_string *pip_make_string()
{
   struct pip_string *s;

   if (( s = ( struct pip_string *)pip_memory( sizeof( struct pip_string ))) == NULL )
      return NULL;

   if (( s->str = ( char *)pip_memory( pip_stack_inc + 1 )) == NULL )
   {
      free( s );
      return NULL;
   }

   *s->str = '\0';
   s->free = pip_stack_inc;
   s->used = 0;
   s->top = s->str;

   return s;
}

void pip_string_free( struct pip_string *s )
{
   free( s->str );
   free( s );
}

void pip_string_append( struct pip_string *s, char c )
{
   pip_alloc_err = 0;

   if ( s->free == 0 )
   {
      s->str = ( char *)realloc( s->str, s->used + 1 + pip_stack_inc );

      if ( s->str == NULL )
      {
         if ( pip_logging )
            syslog( LOG_WARNING, "realloc(): %m" );
         else
            fprintf( stderr, "realloc(): %s\n", strerror( errno ));

         pip_alloc_err = 1;
         return;
      }

      /* Leave room for end-of-string sentinel */

      s->free = pip_stack_inc;
      s->top = &s->str[ s->used ];
   }

   ++s->used;
   --s->free;
   *s->top++ = c;
   *s->top = '\0';
}

#define STRING_APPEND( _s_, _c_ ) { if (( _s_ )->free )\
   {\
      --( _s_ )->free;\
      ++( _s_ )->used;\
      *( _s_ )->top++ = _c_;\
      *( _s_ )->top ='\0';\
   }\
   else pip_string_append( _s_, _c_ ); }

void pip_string_truncate( struct pip_string *s )
{
   if ( s->used )
   {
      s->free += s->used;
      s->used = 0;
      *s->str = '\0';
      s->top = s->str;
   }
}

void pip_set_name( char *name )
{
   char *ptr1, *ptr2;
   int len;

   if ( name == NULL )
      return;

   for( len = 0, ptr1 = name; *ptr1; ++ptr1 )
      ++len;

   pip_app_name = name;

   if (( ptr2 = pip_pidfile = pip_memory( 14 + len )) == NULL )
      exit( 1 );

   for( ptr1 = "/var/run/"; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = name; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = ".pid"; *ptr1; )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';
}

void pip_delete_timer()
{
   struct kevent *kev;
   int n;

   if ( pip_in >= pip_qlen )
      return;

   kev = &pip_inqueue[ pip_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = 0;
   kev->flags = EV_DELETE;
   kev->udata = NULL;

   for( n = 0; n < pip_out; ++n )
      if ( pip_outqueue[ n ].ident == 1 && pip_outqueue[ n ].filter == EVFILT_TIMER )
         pip_outqueue[ n ].ident = 0;
}

void pip_set_timer()
{
   struct kevent *kev;

   if ( pip_in >= pip_qlen )
      return;

   kev = &pip_inqueue[ pip_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = pip_timer * 1000;
   kev->flags = EV_ADD;
   kev->udata = NULL;
}

void pip_set_periodic( void ( *callback )(), int seconds )
{
   if (( pip_timer = seconds ) <= 0 || callback == NULL )
      pip_timer = 0;

   pip_delete_timer();

   if ( pip_timer )
      pip_set_timer();

   pip_periodic = callback;
}

void pip_ev_set( int desc, short filter, u_short flags, struct pip_ccb *item )
{
   struct kevent *kev;

   if ( pip_in >= pip_qlen )
      return;

   kev = &pip_inqueue[ pip_in++ ];

   kev->ident = desc;
   kev->filter = filter;
   kev->fflags = 0;
   kev->flags = flags;
   kev->udata = item;
}

void pip_non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }
}

void pip_signal_handler( int signo )
{
   pip_killed = 1;

   if ( pip_testing )
   {
      pip_exit_func();
      exit( 0 );
   }
}

void pip_set_options( int argc, char **argv )
{
   int i, specified = 0;

   while(( i = getopt( argc, argv, "xr:l:m:u:g:p:i:f:b:")) != -1 )
      switch( i )
      {
         case 'b':
            pip_max_body = strtol( optarg, NULL, 10 );
            break;

         case 'f':
            pip_config_file = optarg;
            break;

         case 'i':
            ++specified;
            pip_interface = optarg;
            break;

         case 'm':
            pip_max_conn = strtol( optarg, NULL, 10 );
            break;

         case 'p':
            ++specified;
            pip_port = optarg;
            break;

         case 'r':
            pip_root_dir = optarg;
            break;

         case 'u':
            pip_user = optarg;
            break;

         case 'g':
            pip_group = optarg;
            break;

         case 'l':
            pip_listen_unix = optarg;
            break;

         case 'x':
            pip_testing = 1;
      }

   if ( specified && pip_listen_unix != NULL )
   {
      fprintf( stderr,
               "%s: the -l option cannot be present when either or both of the -i and -p options are present\n",
               pip_app_name );
      exit( 1 );
   }

   if ( pip_root_dir == NULL || ! *pip_root_dir )
   {
      fprintf( stderr, "%s: root directory undefined (-r option)\n", pip_app_name );
      exit( 1 );
   }

   if ( pip_root_dir != NULL )
   {
      if ( chdir( pip_root_dir ) < 0 )
      {
         fprintf( stderr, "%s: chdir( %s ): %s\n", pip_app_name, pip_root_dir, strerror( errno ));
         exit( 1 );
      }
   }

   if ( pip_max_conn <= 0 )
   {
      fprintf( stderr, "%s: -m max conn value <= 0: %d\n", pip_app_name, pip_max_conn );
      exit( 1 );
   }

   pip_qlen = pip_max_conn * MAX_EVENT;

   if (( pip_passwd = getpwnam( pip_user )) == NULL )
   {
      fprintf( stderr, "%s: user \"%s\" does not exist\n", pip_app_name, pip_user );
      exit( 1 );
   }

   if (( pip_grp = getgrnam( pip_group )) == NULL )
   {
      fprintf( stderr, "%s: group \"%s\" does not exist\n", pip_app_name, pip_group );
      exit( 1 );
   }

   if (( pip_inqueue = pip_memory( sizeof( struct kevent ) * pip_qlen )) == NULL )
      exit( 1 );

   if (( pip_outqueue = pip_memory( sizeof( struct kevent ) * pip_qlen )) == NULL )
      exit( 1 );
}

void pip_become_daemon()
{
   char buffer[ 16 ];
   int len, file;

   /*
    * Fork and let the parent die, continuing as child so we are not
    * a process group leader.  This is necessary for the call to setsid().
    */

   switch( fork() )
   {
      case -1:
         fprintf( stderr, "%s: fork(): %s\n", pip_app_name, strerror( errno ));
         exit( 1 );

      case 0:
         break;

      default:
         exit( 0 );
   }

   fclose( stdout );
   fclose( stderr );
   fclose( stdin );

   stdin = fopen( "/dev/null", "r" );
   stdout = fopen( "/dev/null", "w" );
   stderr = fopen( "/dev/null", "w" );

   if ( stdin == NULL || stdout == NULL || stderr == NULL )
   {
      syslog( LOG_ERR, "fopen(): %m" );
      exit( 1 );
   }

   /*
    * Detach us from our controlling terminal, so job control and other
    * signals may not be sent to us from that terminal.
    */

   if ( setsid() < 0 )
   {
      syslog( LOG_ERR, "setsid(): %m" );
      exit( 1 );
   }

   /*
    * Write our pid to disk.
    */

   if (( file = open( pip_pidfile, O_WRONLY | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP )) < 0 )
   {
      syslog( LOG_WARNING, "open(): %m" );
      return;
   }

   len = snprintf( buffer, sizeof( buffer ), "%d", getpid() );
   write( file, buffer, len );
   close( file );
}

void pip_change_identity()
{
   if (( pip_passwd = getpwnam( pip_user )) == NULL )
   {
      syslog( LOG_ERR, "%s: user \"%s\" does not exist\n", pip_app_name, pip_user );
      exit( 1 );
   }

   if (( pip_grp = getgrnam( pip_group )) == NULL )
   {
      syslog( LOG_ERR, "%s: group \"%s\" does not exist\n", pip_app_name, pip_group );
      exit( 1 );
   }

   if ( setgid( pip_grp->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "setgid(): %m" );
      exit( 1 );
   }

   if ( setuid( pip_passwd->pw_uid ) < 0 )
   {
      syslog( LOG_ERR, "setuid(): %m" );
      exit( 1 );
   }
}

void pip_close_on_exec( int fd )
{
   if ( fcntl( fd, F_SETFD, FD_CLOEXEC ) < 0 )
      syslog( LOG_ERR, "fcntl( F_SETFD, FD_CLOEXEC ): %m" );
}

void pip_start_listening_unix()
{
   struct sockaddr_un sa;

   if (( pip_fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   unlink( pip_listen_unix );
   bzero( &sa, sizeof( struct sockaddr_un ));
   sa.sun_family = AF_UNIX;
   strncpy( sa.sun_path, pip_listen_unix, sizeof( sa.sun_path ) - 1 );  /* ensures NUL-terminated. */

   if ( bind( pip_fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )))
   {
      syslog( LOG_ERR, "bind( %s ): %m", pip_listen_unix );
      close( pip_fd );
      exit( 1 );
   }

   if ( chown( pip_listen_unix, pip_passwd->pw_uid, pip_grp->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "chown( %s ): %m", pip_listen_unix );
      close( pip_fd );
      exit( 1 );
   }

   if ( chmod( pip_listen_unix, S_IRWXU | S_IRWXG ) < 0 )
   {
      syslog( LOG_ERR, "chmod( %s, S_IRWXU | S_IRWXG ): %m", pip_listen_unix );
      close( pip_fd );
      exit( 1 );
   }

   if ( listen( pip_fd, pip_backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      close( pip_fd );
      exit( 1 );
   }

   pip_close_on_exec( pip_fd );
   pip_non_blocking( pip_fd );
}

void pip_start_listening()
{
   struct addrinfo hints, *res;
   int result;

   if ( pip_listen_unix != NULL )
   {
      pip_start_listening_unix();
      return;
   }

   /*
    * This is the new, protocol-independent way of setting up a listening
    * socket.
    */

   bzero( &hints, sizeof( struct addrinfo ));
   hints.ai_flags = AI_PASSIVE;
   hints.ai_socktype = SOCK_STREAM;

   /*
    * If the user has not specified an interface, we listen on the
    * IPv6 wildcard address.
    */

   if (( result = getaddrinfo( ( *pip_interface ? pip_interface : NULL ), pip_port, &hints, &res )))
   {
      syslog( LOG_ERR, "getaddrinfo(): %s", gai_strerror( result ));
      exit( 1 );
   }

   if ( res == NULL )
   {
      syslog( LOG_ERR, "getaddrinfo(): no interface found" );
      exit( 1 );
   }

   pip_fd = socket( res->ai_family, res->ai_socktype, res->ai_protocol );

   if ( pip_fd == -1 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   result = 1;

   /*
    * Allow duplicate bindings, so we can restart immediately.
    */

   if ( setsockopt( pip_fd, SOL_SOCKET, SO_REUSEPORT, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_REUSEPORT ): %m" );

   /*
    * Try and detect connections which go idle for long periods of time.
    */

   if ( setsockopt( pip_fd, SOL_SOCKET, SO_KEEPALIVE, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( SO_KEEPALIVE ): %m" );

   /*
    * Make sure we can accept IPv4 traffic as well as IPv6, if we
    * are bound to the IPv6 wildcard address.  We are turning off
    * the option because result = 0.
    */

   result = 0;

   if ( ! *pip_interface && setsockopt( pip_fd, IPPROTO_IPV6, IPV6_BINDV6ONLY, &result, sizeof( result )) < 0 )
      syslog( LOG_WARNING, "setsockopt( IPV6_BINDV6ONLY ): %m" );

   if ( bind( pip_fd, res->ai_addr, res->ai_addrlen ) < 0 )
   {
      syslog( LOG_ERR, "bind(): %m" );
      exit( 1 );
   }

   freeaddrinfo( res );

   if ( listen( pip_fd, pip_backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      exit( 1 );
   }

   pip_close_on_exec( pip_fd );
   pip_non_blocking( pip_fd );
}

void pip_add_conn( int new )
{
   struct pip_ccb *ptr;

   if (( ptr = pip_memory( sizeof( struct pip_ccb ))) == NULL )
   {
      close( new );
      return;
   }

   bzero( ptr, sizeof( struct pip_ccb ));
   ptr->sock = new;

   ++pip_active;
   pip_ev_set( ptr->sock, EVFILT_READ, EV_ADD, ptr );
}

void pip_remove_conn( struct pip_ccb *item )
{
   struct pip_buffer *ptr, *ptr2;
   char **ptr3;

   for( ptr = item->queue; ptr != NULL; ptr = ptr2 )
   {
      ptr2 = item->queue->next;
      free( ptr->data );
      free( ptr );
   }

   for( ptr3 = item->params; *ptr3 != NULL; ++ptr3 )
      free( *ptr3 );

   if ( item->header != NULL )
      free( item->header );

   if ( item->body != NULL )
      free( item->body );

   if ( item->cookie != NULL )
      free( item->cookie );

   if ( item->query != NULL )
      free( item->query );

   pip_closed = item->sock;
   close( item->sock );
   free( item );

   if ( ! --pip_active && pip_killed )
   {
      pip_exit_func();
      exit( 0 );
   }
}

void pip_accept_connection()
{
   int conn;

   while(( conn = accept( pip_fd, NULL, 0 )) > 0 )
   {
      if ( pip_active == pip_max_conn )
      {
         close( conn );
         return;
      }

      pip_close_on_exec( conn );
      pip_add_conn( conn );
   }

   if ( conn < 0 && errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
   {
      syslog( LOG_ERR, "accept(): %m" );
      exit( 1 );
   }
}

int pip_grow_queue( struct pip_ccb *item )
{
   struct pip_buffer *ptr;

   if (( ptr = pip_memory( sizeof( struct pip_buffer ))) == NULL )
      return 1;

   if ( ! item->qlen )
   {
      item->last = item->queue = ptr;
      pip_ev_set( item->sock, EVFILT_WRITE, EV_ADD, item );
   }
   else
   {
      item->last->next = ptr;
      item->last = item->last->next;
   }

   ++item->qlen;
   item->last->next = NULL;

   return 0;
}

union pip_user_data *pip_get_user_data( void *i )
{
   struct pip_ccb *item;

   if (( item = ( struct pip_ccb *)i ) == NULL )
      return NULL;

   return &item->user_data;
}

int pip_write_conn( void *i, void *buffer, int len )
{
   struct pip_ccb *item;
   int mylen;
   char *ptr, *ptr1, *ptr2;

   if (( item = ( struct pip_ccb *)i ) == NULL )
      return -1;

   if ( len <= 0 )
      return 0;

   if (( ptr1 = ptr = pip_memory( len )) == NULL )
      return -2;

   if ( pip_grow_queue( item ))
   {
      free( ptr );
      return -2;
   }

   ptr2 = buffer;

   for( mylen = len; mylen; --mylen )
      *ptr1++ = *ptr2++;

   item->last->total = len;
   item->last->data = ptr;

   return 0;
}

char *pip_find_next( struct pip_ccb *item, char *ptr, int *len )
{
   while( --*len )
      if ( ! *ptr++ )
         break;

   if ( ! *len )
      return NULL;

   return ptr;
}

int pip_init_env( struct pip_ccb *item, int len )
{
   char *ptr;
   int n;

   n = 0;

   for( ptr = item->header; ptr != NULL; ptr = pip_find_next( item, ptr, &len ))
   {
      item->env[ n++ ] = ptr;

      if ( n == MAX_ENV )
         break;
   }

   if ( n % 2 )
      return 1;

   item->env[ n ] = NULL;
   pip_process_cookies( item );

   return 0;
}

char **pip_lookup_list( struct pip_ccb *item, char **list, char *key )
{
   char **ptr, *ptr1, *ptr2;

   if ( list == NULL )
      return NULL;

   for( ptr = list; *ptr != NULL; ptr += 2 )
   {
      ptr1 = *ptr;
      ptr2 = key;

      while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
      {
         ++ptr1;
         ++ptr2;
      }

      if ( ! *ptr1 && ! *ptr2 )
         return &ptr[ 1 ];
   }

   return NULL;
}

char *pip_get_env( void *i, char *key )
{
   struct pip_ccb *item;
   char **ptr;

   if ( key == NULL || ( item = ( struct pip_ccb *)i ) == NULL )
      return NULL;

   if (( ptr = pip_lookup_list( item, item->env, key )) == NULL )
      return NULL;

   return *ptr;
}

char **pip_get_envs( void *i )
{
   struct pip_ccb *item;

   if (( item = ( struct pip_ccb *)i ) == NULL )
      return NULL;

   return item->env;
}

char *pip_html_escape( char *str )
{
   struct pip_string *s;
   char *ptr;

   if (( s = pip_make_string()) == NULL )
      return NULL;

   for( ptr = str; *ptr; ++ptr )
      switch( *ptr )
      {
         case '&':
            STRING_APPEND( s, '&' )
            STRING_APPEND( s, 'a' )
            STRING_APPEND( s, 'm' )
            STRING_APPEND( s, 'p' )
            STRING_APPEND( s, ';' )
            if ( pip_alloc_err )
               return NULL;
            break;

         case '<':
            STRING_APPEND( s, '&' )
            STRING_APPEND( s, 'l' )
            STRING_APPEND( s, 't' )
            STRING_APPEND( s, ';' )
            if ( pip_alloc_err )
               return NULL;
            break;

         case '>':
            STRING_APPEND( s, '&' )
            STRING_APPEND( s, 'g' )
            STRING_APPEND( s, 't' )
            STRING_APPEND( s, ';' )
            if ( pip_alloc_err )
               return NULL;
            break;

         default:
            STRING_APPEND( s, *ptr )
            if ( pip_alloc_err )
               return NULL;
      }

   ptr = s->str;
   free( s );
   return ptr;
}

char *pip_form_decode( char *str )
{
   struct pip_string *s;
   char *ptr;

   if (( s = pip_make_string()) == NULL )
      return NULL;

   for( ptr = str; *ptr; ++ptr )
      switch( *ptr )
      {
         case '+':
            STRING_APPEND( s, ' ' )
            if ( pip_alloc_err )
               return NULL;
            break;

         case '%':
            if ( ! *( ptr + 1 ) || ! *( ptr + 2 ))
            {
               STRING_APPEND( s, *ptr )
               if ( pip_alloc_err )
                  return NULL;
            }
            else
            {
               char data[ 3 ];
               char code;

               data[ 0 ] = *( ptr + 1 );
               data[ 1 ] = *( ptr + 2 );
               data[ 2 ] = '\0';

               if (( code = ( char )strtol( data, NULL, 16 )))
               {
                  STRING_APPEND( s, code )
                  if ( pip_alloc_err )
                     return NULL;

                  ptr += 2;
               }
               else
               {
                  STRING_APPEND( s, *ptr )
                  if ( pip_alloc_err )
                     return NULL;
               }
            }
            break;

         default:
            STRING_APPEND( s, *ptr )
            if ( pip_alloc_err )
               return NULL;
      }

   ptr = s->str;
   free( s );
   return ptr;
}

char *pip_form_encode( char *str )
{
   char *ptr, *ptr2;
   struct pip_string *s = NULL;
   int found;
   static char *reserved = "']$;:@&+!?=#{}/[^`~\"<>|%\\\t\r\n",
               *letters  = "0123456789ABCDEF";

   if (( s = pip_make_string()) == NULL )
      return NULL;

   for( ptr = str; *ptr; ++ptr )
   {
      found = 0;

      if ( *ptr == ' ' )
      {
         STRING_APPEND( s, '+' )
         if ( pip_alloc_err )
            return NULL;

         continue;
      }

      if ( *ptr < 32 || *ptr > 126 )
      {
         STRING_APPEND( s, '%' )
         STRING_APPEND( s, letters[ *ptr / 16 ] )
         STRING_APPEND( s, letters[ *ptr % 16 ] )
         if ( pip_alloc_err )
            return NULL;

         continue;
      }

      for( ptr2 = reserved; *ptr2; ++ptr2 )
         if ( *ptr == *ptr2 )
         {
            STRING_APPEND( s, '%' )
            STRING_APPEND( s, letters[ *ptr / 16 ] )
            STRING_APPEND( s, letters[ *ptr % 16 ] )
            if ( pip_alloc_err )
               return NULL;

            found = 1;
            break;
         }

      if ( ! found )
      {
         STRING_APPEND( s, *ptr )
         if ( pip_alloc_err )
            return NULL;
      }
   }

   ptr = s->str;
   free( s );
   return ptr;
}

void pip_process_cookies( struct pip_ccb *item )
{
   char *cookie, *name, *ptr;
   int i;

   if (( cookie = pip_get_env( item, "HTTP_COOKIE" )) == NULL || ! *cookie )
      return;

   if (( item->cookie = pip_str_dup( cookie, -1 )) == NULL )
   {
      pip_remove_conn( item );
      return;
   }

   cookie = item->cookie;

   for( i = 0, ptr = cookie; *ptr; ++ptr )
   {
      if ( *ptr == ';' || *ptr == ',' )
      {
         if ( i >= MAX_COOKIE )
            break;

         *ptr = '\0';

         for( name = cookie; *name && *name != '='; ++name )
            ;

         if ( *name )
            *name++ = '\0';

         while( *cookie == ' ' )
            ++cookie;

         item->cookies[ i++ ] = cookie;
         item->cookies[ i++ ] = name;

         cookie = ptr + 1;
      }
   }

  if ( *cookie && i < MAX_COOKIE - 1 )
  {
      for( name = cookie; *name && *name != '='; ++name )
         ;

      if ( *name )
         *name++ = '\0';

      while( *cookie == ' ' )
         ++cookie;

      item->cookies[ i++ ] = cookie;
      item->cookies[ i++ ] = name;
   }
}

void pip_process_params( struct pip_ccb *item )
{
   char *query, *ptr, *param;
   int i, flag = 0;

   query = NULL;

   if (( query = pip_get_env( item, "QUERY_STRING" )) == NULL || ! *query )
   {
      if (( query = item->body ) == NULL || ! *query )
         return;
   }
   else
   {
      ++flag;

      /*
       * item->query is freed pip_remove_conn().
       * The original query is freed when item->params is freed in pip_remove_conn().
       */
      
      if (( item->query = pip_str_dup( query, -1 )) == NULL )
      {
         pip_remove_conn( item );
         return;
      }

      query = item->query;
   }

   i = 0;

MORE:
   for( ptr = query; *ptr; ++ptr )
   {
      if ( *ptr == '&' )
      {
         if ( i >= MAX_PARAM )
            break;

         *ptr = '\0';

         for( param = query; *param && *param != '='; ++param )
            ;

         if ( *param )
            *param++ = '\0';

         item->params[ i++ ] = pip_form_decode( query );
         item->params[ i++ ] = pip_form_decode( param );

         query = ptr + 1;
      }
   }

  if ( *query && i < MAX_PARAM - 1 )
  {
      for( param = query; *param && *param != '='; ++param )
         ;

      if ( *param )
         *param++ = '\0';

      item->params[ i++ ] = pip_form_decode( query );
      item->params[ i++ ] = pip_form_decode( param );
   }

   if ( flag && item->body != NULL )
   {
      query = item->body;
      --flag;
      goto MORE;
   }
}

char *pip_get_param( void *i, char *param )
{
   struct pip_ccb *item;
   char **ptr;

   if ( param == NULL || ( item = ( struct pip_ccb *)i ) == NULL )
      return NULL;

   if (( ptr = pip_lookup_list( item, item->params, param )) == NULL )
      return NULL;

   return *ptr;
}

char **pip_get_params( void *i )
{
   struct pip_ccb *item;

   if (( item = ( struct pip_ccb *)i ) == NULL )
      return NULL;

   return item->params;
}

char *pip_get_cookie( void *i, char *name )
{
   struct pip_ccb *item;
   char **ptr;

   if ( name == NULL || ( item = ( struct pip_ccb *)i ) < 0 )
      return NULL;

   if (( ptr = pip_lookup_list( item, item->cookies, name )) == NULL )
      return NULL;

   return *ptr;
}

char **pip_get_cookies( void *i )
{
   struct pip_ccb *item;

   if (( item = ( struct pip_ccb *)i ) < 0 )
      return NULL;

   return item->cookies;
}

void pip_invoke_handler( struct pip_ccb *item )
{
   if ( pip_request_handler( item ))
   {
      pip_remove_conn( item );
      return;
   }

   if ( ! item->qlen )
      pip_remove_conn( item );
}

int pip_read_header( struct pip_ccb *item )
{
   char *ptr, c;
   int n;

   if ( item->count >= 0 )
   {
      for( ptr = &item->inbuf[ item->count ]; item->count < sizeof( item->inbuf ); ++ptr, ++item->count )
      {
         if (( n = read( item->sock, &c, 1 )) <= 0 )
         {
            if ( n < 0 && errno == EWOULDBLOCK )
               return 1;

            return -1;
         }

         if ( c == ':' )
         {
            item->count = 0;
            *ptr = '\0';
            break;
         }

         *ptr = c;
      }

      if ( c != ':' )
         return -1;
   }

   /*
    * item->count will be only zero here immediately after we exit the
    * preceding loop via the break statement.
    */

   if ( ! item->count )
   {
      if (( item->count = strtol( item->inbuf, NULL, 10 )) <= 0 )
         return -1;

      if (( item->header = pip_memory( item->count + 1 )) == NULL )
         return -1;

      item->count = -( item->count + 1 );
   }

   /*
    * item->count will be negative while we read the header.
    */

   if ( item->count < 0 )
   {
      if (( n = read( item->sock, &item->header[ item->offset ], -item->count )) <= 0 )
      {
         if ( n < 0 && errno == EWOULDBLOCK )
            return 1;

         return -1;
      }

      item->count += n;
      item->offset += n;

      if ( item->count )
         return 1;
   }

   if ( item->header[ item->offset - 1 ] != ',' )
   {
      pip_remove_conn( item );
      return -1;
   }

   if ( pip_init_env( item, item->offset - 1 ))
      return -1;

   item->offset = item->count = 0;
   return 0;
}

int pip_detect_body( struct pip_ccb *item )
{
   char *ptr, *ptr1, *ptr2;

   if (( ptr = pip_get_env( item, "CONTENT_LENGTH" )) == NULL )
      return -1;

   if ( !( item->count = strtol( ptr, NULL, 10 )))
      return 0;

   if ( item->count < 0 || item->count > pip_max_body )
      return -1;

   if (( ptr = pip_get_env( item, "CONTENT_TYPE" )) == NULL )
      return 0;

   ptr1 = "multipart/form-data";
   ptr2 = ptr;

   while( *ptr1 && *ptr2 && *ptr1 == *ptr2 )
   {
      ++ptr1;
      ++ptr2;
   }

   if ( ! *ptr1 )
      return -1;

   if (( item->body = pip_memory( item->count + 1 )) == NULL )
      return -1;

   return 1;
}

void pip_read_body( struct pip_ccb *item )
{
   int n;

   if (( n = read( item->sock, &item->body[ item->offset ], item->count )) <= 0 )
   {
      if ( n < 0 && errno == EWOULDBLOCK )
         return;

      pip_remove_conn( item );
      return;
   }

   item->offset += n;
   item->count -= n;

   if ( item->count )
      return;

   item->body[ item->offset ] = '\0';
   item->offset = item->count = 0;

   pip_process_params( item );
   pip_ev_set( item->sock, EVFILT_READ, EV_DELETE, item );
   pip_invoke_handler( item );
}

void pip_transfer_in( struct pip_ccb *item )
{
   switch( pip_read_header( item ))
   {
      case -1:
         pip_remove_conn( item );
         return;

      case 0:
         break;

      case 1:
         return;
   }

   switch( pip_detect_body( item ))
   {
      case -1:
         pip_remove_conn( item );
         return;

      case 0:
         break;

      default:
         ++item->state;
         return;
   }

   pip_process_params( item );
   pip_ev_set( item->sock, EVFILT_READ, EV_DELETE, item );
   pip_invoke_handler( item );
}

void pip_shrink_queue( struct pip_ccb *item )
{
   struct pip_buffer *ptr;

   free( item->queue->data );
   ptr = item->queue;
   item->queue = item->queue->next;
   free( ptr );

   item->count = 0;

   if ( ! --item->qlen )
      pip_remove_conn( item );
}

void pip_transfer_out( struct pip_ccb *item )
{
   off_t count;

   if ( ! item->qlen )
   {
      pip_remove_conn( item );
      return;
   }

   count = write( item->sock, &item->queue->data[ item->count ], item->queue->total );

   if ( count < 0 )
   {
      if ( errno != EWOULDBLOCK )
         pip_remove_conn( item );

      return;
   }

   item->count += count;
   item->queue->total -= count;

   if ( ! item->queue->total )
      pip_shrink_queue( item );
}

void pip_set_sigterm_intr()
{
   struct sigaction sigact;

   sigact.sa_handler = pip_signal_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGTERM, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction: %s\n", strerror( errno ));
}

/*
 * If we have dropped a connection, we traverse the output kqueue
 * and mark any events for that connection as invalid, by setting
 * the socket descriptor to 0.
 */

void pip_remove_events( int n )
{
   while( ++n < pip_out )
      if ( pip_outqueue[ n ].ident == pip_closed )
         pip_outqueue[ n ].ident = 0;
}

void pip_process_clients()
{
   int kq, n, killed = 0;

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   pip_ev_set( pip_fd, EVFILT_READ, EV_ADD, NULL );

   for( ; ; )
   {
      pip_set_sigterm_intr();
      pip_out = kevent( kq, pip_inqueue, pip_in, pip_outqueue, pip_qlen, NULL );
      pip_in = 0;

      if ( pip_killed && ! killed )
      {
         if ( ! pip_active )
         {
            pip_exit_func();
            exit( 0 );
         }

         close( pip_fd );
         ++killed;
         continue;
      }

      if ( pip_out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %m" );
      }

      signal( SIGTERM, pip_signal_handler );

      for( n = 0; n < pip_out; ++n )
      {
         struct pip_ccb *item = ( struct pip_ccb *)pip_outqueue[ n ].udata;

         if ( ! pip_outqueue[ n ].ident || pip_outqueue[ n ].flags & EV_ERROR )
            continue;

         pip_closed = 0;

         if ( pip_outqueue[ n ].ident == pip_fd )
         {
            if ( ! pip_killed )
               pip_accept_connection();
         }
         else if ( pip_outqueue[ n ].filter == EVFILT_TIMER )
         {
            if ( pip_periodic != NULL )
               pip_periodic();
         }
         else if ( pip_outqueue[ n ].filter == EVFILT_WRITE )
            pip_transfer_out( item );
         else if ( ! item->state )
            pip_transfer_in( item );
         else if ( item->count )
            pip_read_body( item );
         else
            pip_remove_conn( item );

         if ( pip_closed )
            pip_remove_events( n );
      }
   }
}

void pip_set_signals()
{
   int *iptr;
   int sigs[] = {
      SIGPIPE, SIGHUP, SIGQUIT, SIGUSR1, SIGUSR2, SIGALRM, SIGINT, SIGTSTP, -1
   };

   /*
    * We ignore everything but SIGTERM, SIGBUS, and SIGSEGV when ! testing.
    */

   signal( SIGTERM, pip_signal_handler );

   for( iptr = sigs; *iptr > 0; ++iptr )
      signal( *iptr, ( pip_testing ? pip_signal_handler : SIG_IGN ));
}

int main( int argc, char **argv )
{
   pip_set_options( argc, argv );
   pip_set_signals();

   pip_init_func();
   openlog( pip_app_name, LOG_PID, LOG_DAEMON );
   pip_logging = 1;

   if ( ! pip_testing )
      pip_become_daemon();

   pip_start_listening();
   pip_change_identity();
   pip_process_clients();

   return 0;
}
