/*
 * LibDorrit Copyright (c) 2015-2018, James Bailie.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *     * The name of James Bailie may not be used to endorse or promote
 * products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/event.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/un.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

#include <pwd.h>
#include <grp.h>

#include <syslog.h>
#include <stdarg.h>
#include <string.h>

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

/*
 * 65535 max frame length + 4 for header + 1 for terminator.
 */

#define IOBUFSIZE 65540

/*
 * KQueue Input and output queues.
 */

#define QLEN 16384
#define MAX_EVENT 10

struct kevent *ws_inqueue, *ws_outqueue;

extern char *optarg;

int ws_idx = 0, ws_in = 0, ws_out = 0, ws_fd = -1, ws_logging = 0, ws_testing = 0, ws_timer = 0, ws_alloc_err = 0,
    ws_user_socket = 0, ws_max_conn = QLEN, ws_qlen = QLEN * MAX_EVENT;

unsigned int ws_stack_inc = 128, ws_backlog = 1000, ws_conn_count = 0, ws_buffer_size = IOBUFSIZE;

char *ws_grp = "nobody", *ws_user = "nobody", *ws_listen_unix = "/var/run/dorrit.socket",
     *ws_app_name = "dorrit", *ws_pidfile = "/var/run/dorrit.pid", *ws_config_file = NULL;

void ( *ws_periodic )() = NULL;

struct passwd *ws_passwd;
struct group *ws_group;

volatile int ws_sigterm = 0;

/*
 * Queue of buffered out-going data for each connection.
 */

struct ws_buffer
{
   unsigned int refcount;
   char *buffer;
};

struct ws_queue
{
   unsigned int count, total;
   struct ws_buffer *buffer;
   struct ws_queue *next;
};

/*
 * Connection control block.
 */

struct ws_ccb
{
   int sock, stage;
   unsigned int len, total, offset, qlen, closed;
   struct ws_queue *first, *last;
   struct ws_string *cookie;
   void *data;
   unsigned char *buffer, mask[ 4 ];
};

struct ws_string
{
   int free, used;
   char *top;
   char *str;
};

void ws_set_name( char * );
void ws_set_periodic( void (*)(), int );

int ws_write_conn( struct ws_ccb **, int, int, unsigned int, unsigned char * );
void ws_close_conn( struct ws_ccb *, int );

void ws_init_func();
void ws_exit_func();

int ws_open_callback( void *, char * );
void ws_close_callback( void * );
void ws_read_callback( void *, int, int, unsigned char * );

void *ws_memory( int size )
{
   void *ptr;

   if ( size == 0 )
      return NULL;

   if (( ptr = malloc( size )) == NULL )
   {
      if ( ws_logging )
         syslog( LOG_WARNING, "malloc(): %m" );
      else
         fprintf( stderr, "dorrit: malloc(): %s\n", strerror( errno ));

      return NULL;
   }

   return ptr;
}

void ws_set_name( char *name )
{
   char *ptr1, *ptr2;
   int len;

   if ( name == NULL )
      return;

   for( len = 0, ptr1 = name; *ptr1; ++ptr1 )
      ++len;

   ws_app_name = name;

   if (( ptr2 = ws_pidfile = ws_memory( 14 + len )) == NULL )
      exit( 1 );

   for( ptr1 = "/var/run/"; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = name; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = ".pid"; *ptr1; )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';

   if ( ws_user_socket )
      return;

   if (( ptr2 = ws_listen_unix = ws_memory( 17 + len )) == NULL )
      exit( 1 );

   for( ptr1 = "/var/run/"; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = name; *ptr1; )
      *ptr2++ = *ptr1++;

   for( ptr1 = ".socket"; *ptr1; )
      *ptr2++ = *ptr1++;

   *ptr2 = '\0';
}

struct ws_string *ws_make_string()
{
   struct ws_string *s;

   if (( s = ( struct ws_string *)ws_memory( sizeof( struct ws_string ))) == NULL )
      return NULL;

   if (( s->str = ( char *)ws_memory( ws_stack_inc + 1 )) == NULL )
   {
      free( s );
      return NULL;
   }

   *s->str = '\0';
   s->free = ws_stack_inc;
   s->used = 0;
   s->top = s->str;

   return s;
}

void ws_string_free( struct ws_string *s )
{
   free( s->str );
   free( s );
}

void ws_string_append( struct ws_string *s, char c )
{
   ws_alloc_err = 0;

   if ( s->free == 0 )
   {
      s->str = ( char *)realloc( s->str, s->used + 1 + ws_stack_inc );

      if ( s->str == NULL )
      {
         if ( ws_logging )
            syslog( LOG_WARNING, "realloc(): %m" );
         else
            fprintf( stderr, "realloc(): %s\n", strerror( errno ));

         ws_alloc_err = 1;
         return;
      }

      /* Leave room for end-of-string sentinel */

      s->free = ws_stack_inc;
      s->top = &s->str[ s->used ];
   }

   ++s->used;
   --s->free;
   *s->top++ = c;
   *s->top = '\0';
}

#define STRING_TRUNCATE( _s_ ) if (( _s_ )->used )\
  {\
     ( _s_ )->free += ( _s_ )->used;\
     ( _s_ )->used = 0;\
     *( _s_ )->str = '\0';\
     ( _s_ )->top = ( _s_ )->str;\
  }

#define STRING_APPEND( _s_, _c_ ) if (( _s_ )->free )\
   {\
      --( _s_ )->free;\
      ++( _s_ )->used;\
      *( _s_ )->top++ = _c_;\
      *( _s_ )->top ='\0';\
   }\
   else ws_string_append( _s_, _c_ )

void ws_ev_set( int desc, short filter, u_short flags, struct ws_ccb *item )
{
   struct kevent *kev;

   if ( ws_in >= ws_qlen )
      return;

   kev = &ws_inqueue[ ws_in++ ];

   kev->ident = desc;
   kev->filter = filter;
   kev->fflags = 0;
   kev->flags = flags;
   kev->udata = item;
}

void ws_set_timer()
{
   struct kevent *kev;

   if ( ws_in >= ws_qlen )
      return;

   kev = &ws_inqueue[ ws_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = ws_timer * 1000;
   kev->flags = EV_ADD;
   kev->udata = NULL;
}

void ws_delete_timer()
{
   struct kevent *kev;

   if ( ws_in >= ws_qlen )
      return;

   kev = &ws_inqueue[ ws_in++ ];

   kev->ident = 1;
   kev->filter = EVFILT_TIMER;
   kev->fflags = 0;
   kev->data = 0;
   kev->flags = EV_DELETE;
   kev->udata = NULL;
}

void ws_set_periodic( void ( *callback )(), int seconds )
{
   if (( ws_timer = seconds ) <= 0 || callback == NULL )
      ws_timer = 0;

   ws_delete_timer();

   if ( ws_timer )
      ws_set_timer();

   ws_periodic = callback;
}

void ws_non_blocking( int desc )
{
   int flags, unblocked;

   if (( flags = fcntl( desc, F_GETFL, 0 )) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }

   unblocked = flags & O_NONBLOCK;

   if ( ! unblocked && fcntl( desc, F_SETFL, flags | O_NONBLOCK ) < 0 )
   {
      syslog( LOG_ERR, "fcntl(): %m" );
      exit( 1 );
   }
}

void ws_set_options( int argc, char **argv )
{
   int i;

   while(( i = getopt( argc, argv, "xb:l:m:u:g:f:" )) != -1 )
      switch( i )
      {
         case 'b':
            ws_buffer_size = strtol( optarg, NULL, 10 );
            break;

         case 'f':
            ws_config_file = optarg;
            break;

         case 'g':
            ws_grp = optarg;
            break;

         case 'm':
            ws_max_conn = strtol( optarg, NULL, 10 );
            break;

         case 'u':
            ws_user = optarg;
            break;

         case 'l':
            ws_listen_unix = optarg;
            ++ws_user_socket;
            break;

         case 'x':
            ws_testing = 1;
      }

   if ( ws_max_conn <= 0 )
   {
      fprintf( stderr, "%s: -m <max-conn> <= 0: %d\n", ws_app_name, ws_max_conn );
      exit( 1 );
   }

   if ( ws_buffer_size < 10 || ws_buffer_size > 65540 )
   {
      fprintf( stderr, "%s: (-b) buffer size not within range 10-65540: %d\n", ws_app_name, ws_buffer_size );
      exit( 1 );
   }

   if ( ws_listen_unix == NULL )
   {
      fprintf( stderr, "%s: listening socket unspecified\n", ws_app_name );
      exit( 1 );
   }

   if (( ws_passwd = getpwnam( ws_user )) == NULL )
   {
      fprintf( stderr, "%s: user \"%s\" does not exist\n", ws_app_name, ws_user );
      exit( 1 );
   }

   if (( ws_group = getgrnam( ws_grp )) == NULL )
   {
      fprintf( stderr, "%s: group \"%s\" does not exist\n", ws_app_name, ws_grp );
      exit( 1 );
   }
}

void ws_become_daemon()
{
   int file;

   /*
    * Fork and let the parent die, continuing as child so we are not
    * a process group leader.  This is necessary for the call to setsid().
    */

   switch( fork() )
   {
      case -1:
         fprintf( stderr, "dorrit: fork(): %s\n", strerror( errno ));
         exit( 1 );

      case 0:
         break;

      default:
         exit( 0 );
   }

   fclose( stdout );
   fclose( stderr );
   fclose( stdin );

   stdin = fopen( "/dev/null", "r" );
   stdout = fopen( "/dev/null", "w" );
   stderr = fopen( "/dev/null", "w" );

   if ( stdin == NULL || stdout == NULL || stderr == NULL )
   {
      syslog( LOG_ERR, "fopen(): %m" );
      exit( 1 );
   }

   /*
    * Detach us from our controlling terminal, so job control and other
    * signals may not be sent to us from that terminal.
    */

   if ( setsid() < 0 )
   {
      syslog( LOG_ERR, "setsid(): %m" );
      exit( 1 );
   }

   /*
    * Write our pid to disk.
    */

   if (( file = open( ws_pidfile, O_WRONLY | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP )) < 0 )
      syslog( LOG_WARNING, "open(): %m" );
   else
   {
      char buffer[ 16 ];
      int t;

      t = snprintf( buffer, sizeof( buffer ), "%d", getpid() );
      write( file, buffer, t );
      close( file );
   }

   /*
    * We don't create any files or fork children which create files,
    * but if we did, this will make those files not world-readable by
    * default.
    */

   umask( 2 );
}

void ws_change_identity()
{
   if ( setgid( ws_group->gr_gid ) < 0 )
      syslog( LOG_ERR, "setgid(): %m" );

   if ( setuid( ws_passwd->pw_uid ) < 0 )
      syslog( LOG_ERR, "setuid(): %m" );
}

void ws_close_on_exec( int fd )
{
   if ( fcntl( fd, F_SETFD, FD_CLOEXEC ) < 0 )
      syslog( LOG_ERR, "fcntl( F_SETFD, FD_CLOEXEC ): %m" );
}

void ws_start_listening_unix()
{
   struct sockaddr_un sa;

   if (( ws_fd = socket( PF_LOCAL, SOCK_STREAM, 0 )) < 0 )
   {
      syslog( LOG_ERR, "socket(): %m" );
      exit( 1 );
   }

   unlink( ws_listen_unix );
   bzero( &sa, sizeof( struct sockaddr_un ));
   sa.sun_family = AF_UNIX;
   strncpy( sa.sun_path, ws_listen_unix, sizeof( sa.sun_path ) - 1 );  /* ensures NUL-terminated. */

   if ( bind( ws_fd, ( struct sockaddr *)&sa, SUN_LEN( &sa )))
   {
      syslog( LOG_ERR, "bind( %s ): %m", ws_listen_unix );
      close( ws_fd );
      exit( 1 );
   }

   if ( chown( ws_listen_unix, ws_passwd->pw_uid, ws_group->gr_gid ) < 0 )
   {
      syslog( LOG_ERR, "chown( %s ): %m", ws_listen_unix );
      close( ws_fd );
      exit( 1 );
   }

   if ( chmod( ws_listen_unix, S_IRWXU | S_IRWXG ) < 0 )
   {
      syslog( LOG_ERR, "chmod( %s, S_IRWXU | S_IRWXG ): %m", ws_listen_unix );
      close( ws_fd );
      exit( 1 );
   }

   if ( listen( ws_fd, ws_backlog ) < 0 )
   {
      syslog( LOG_ERR, "listen(): %m" );
      close( ws_fd );
      exit( 1 );
   }

   ws_close_on_exec( ws_fd );
   ws_non_blocking( ws_fd );
}

void ws_set_data( struct ws_ccb *item, void *data )
{
   if ( item != NULL )
      item->data = data;
}

void *ws_get_data( struct ws_ccb *item )
{
   if ( item != NULL )
      return item->data;

   return NULL;
}

unsigned int ws_get_qlen( struct ws_ccb *item )
{
   if ( item != NULL )
      return item->qlen;

   return 0;
}

int ws_queue_pong( struct ws_ccb *item )
{
   struct ws_queue *q;

   if (( q = ws_memory( sizeof( struct ws_queue ))) == NULL )
      return 1;

   if (( q->buffer = ws_memory( sizeof( struct ws_buffer ))) == NULL )
   {
      free( q );
      return 1;
   }

   q->buffer->refcount = 1;

   if (( q->buffer->buffer = ws_memory( item->len + item->offset )) == NULL )
   {
      free( q->buffer );
      free( q );
      return 1;
   }

   ++item->qlen;

   if ( item->first == NULL )
   {
      item->first = item->last = q;
      ws_ev_set( item->sock, EVFILT_WRITE, EV_ENABLE, item );
   }
   else
   {
      item->last->next = q;
      item->last = q;
   }

   item->last->next = NULL;
   item->last->count = 0;
   item->last->total = item->len + item->offset;

   item->last->buffer->buffer[ 0 ] = 0x8a;

   if ( item->offset == 2 )
      item->last->buffer->buffer[ 1 ] = item->len;
   else
   {
      item->last->buffer->buffer[ 1 ] = 126;
      item->last->buffer->buffer[ 2 ] = item->buffer[ 2 ];
      item->last->buffer->buffer[ 3 ] = item->buffer[ 3 ];
   }

   bcopy( &item->buffer[ item->offset ], &item->last->buffer->buffer[ item->offset ], item->len );
   return 0;
}

int ws_queue_each_client( struct ws_ccb **items, int conns, struct ws_buffer *buffer, int len )
{
   struct ws_ccb **ptr;
   struct ws_queue *q;
   int n;

   for( n = 0, ptr = items; n < conns; ++n, ++ptr )
   {
      if (( *ptr )->closed )
         continue;

      if (( q = ws_memory( sizeof( struct ws_queue ))) == NULL )
         return 1;

      q->buffer = buffer;
      ++q->buffer->refcount;
      ++( *ptr )->qlen;

      if (( *ptr )->first == NULL )
      {
         ( *ptr )->first = ( *ptr )->last = q;
         ws_ev_set(( *ptr )->sock, EVFILT_WRITE, EV_ENABLE, *ptr );
      }
      else
      {
         ( *ptr )->last->next = q;
         ( *ptr )->last = q;
      }

      ( *ptr )->last->next = NULL;
      ( *ptr )->last->count = 0;
      ( *ptr )->last->total = len;
   }

   return 0;
}

int ws_write_conn( struct ws_ccb **items, int conns, int binary, unsigned int len, unsigned char *data )
{
   struct ws_buffer *buffer;
   int i, flen, frames;
   unsigned char *payload;

   if ( len < 1 )
      return -2;

   if ( ! ( frames = len / ( ws_buffer_size - 5 )) || len % ( ws_buffer_size - 5 ))
      ++frames;

   payload = data;

   for( i = 0; i < frames; ++i )
   {
      if (( buffer = ws_memory( sizeof( struct ws_buffer ))) == NULL )
         return -1;

      flen = ( len < ( ws_buffer_size - 5 ) ? len + 4 : ws_buffer_size );

      if (( buffer->buffer = ws_memory( flen )) == NULL )
      {
         free( buffer );
         return -1;
      }

      if ( ! i )
         buffer->buffer[ 0 ] = ( frames == 1 ? ( binary ? 0x82 : 0x81 ) : ( binary ? 0x02 : 0x01 ));
      else
         buffer->buffer[ 0 ] = ( i == ( frames - 1 ) ? 0x80 : 0x00 );

      flen = len;

      if ( flen < 126 )
      {
         buffer->buffer[ 1 ] = flen;
         bcopy( payload, &buffer->buffer[ 2 ], flen );
      }
      else
      {
         if ( i != ( frames - 1 ))
         {
            flen = ws_buffer_size - 5;
            len -= flen;
         }

         buffer->buffer[ 1 ] = 126;
         buffer->buffer[ 2 ] = flen / 256;
         buffer->buffer[ 3 ] = flen % 256;

         bcopy( payload, &buffer->buffer[ 4 ], flen );
         payload += flen;
      }

      buffer->refcount = 0;

      if ( ws_queue_each_client( items, conns, buffer, ( flen < 126 ? flen + 2 : flen + 4 )))
      {
         if ( ! buffer->refcount )
         {
            free( buffer->buffer );
            free( buffer );
         }

         return -1;
      }
   }

   return 0;
}

void ws_remove_conn( struct ws_ccb *item )
{
   int i, closed;

   char bye[] = { 0x88, 0 };

   ws_close_callback( item );

   write( item->sock, bye, 2 );
   --ws_conn_count;

   if ( item->buffer != NULL )
      free( item->buffer );

   if ( item->cookie != NULL )
      ws_string_free( item->cookie );

   closed = item->sock;
   close( item->sock );

   while( item->first != NULL )
   {
      item->last = item->first->next;

      if ( ! --item->first->buffer->refcount )
      {
         free( item->first->buffer->buffer );
         free( item->first->buffer );
      }

      free( item->first );
      item->first = item->last;
   }

   free( item );

   for( i = ws_idx + 1; i < ws_out; ++i )
      if ( ws_outqueue[ i ].ident == closed )
         ws_outqueue[ i ].ident = 0;
}

void ws_close_conn( struct ws_ccb *item, int force )
{
   if ( item->qlen && ! force )
      ++item->closed;
   else
      ws_remove_conn( item );
}

void ws_add_conn( int new )
{
   struct ws_ccb *ptr;

   if (( ptr = ws_memory( sizeof( struct ws_ccb ))) == NULL )
   {
      close( new );
      return;
   }

   bzero( ptr, sizeof( struct ws_ccb ));

   if (( ptr->cookie = ws_make_string()) == NULL )
   {
      close( new );
      free( ptr );
      return;
   }

   ws_close_on_exec( new );
   ptr->sock = new;
   ptr->stage = -1;

   ws_ev_set( ptr->sock, EVFILT_READ, EV_ADD | EV_ENABLE, ptr );
   ws_ev_set( ptr->sock, EVFILT_READ, EV_ADD | EV_DISABLE, ptr );

   ++ws_conn_count;
   return;
}

void read_cookie( struct ws_ccb *item )
{
   char c;
   int r;

   while(( r = read( item->sock, &c, 1 )) > 0 )
   {
      if ( c == '\n' )
         break;

      STRING_APPEND( item->cookie, c );

      if ( ws_alloc_err )
      {
         ws_remove_conn( item );
         return;
      }
   }

   if ( ! r )
   {
      ws_remove_conn( item );
      return;
   }

   if ( r < 0 )
   {
      if ( errno == EWOULDBLOCK )
         return;

      ws_remove_conn( item );
      return;
   }

   ++item->stage;

   if ( ws_open_callback( item, item->cookie->str ))
   {
      ws_remove_conn( item );
      return;
   }

   ws_string_free( item->cookie );
   item->cookie = NULL;

   /*
    * Just enough to hold the WebSocket header for now.
    */

   if (( item->buffer = ws_memory( 4 )) == NULL )
   {
      ws_remove_conn( item );
      return;
   }
}

void ws_accept_connection()
{
   int conn;

   while(( conn = accept( ws_fd, NULL, NULL )) > 0 )
   {
      if ( ws_conn_count == ws_max_conn )
      {
         close( conn );
         return;
      }

      ws_add_conn( conn );
   }

   if ( conn < 0 && errno != ECONNABORTED && errno != EWOULDBLOCK && errno != EAGAIN )
   {
      syslog( LOG_ERR, "accept(): %m" );
      exit( 1 );
   }
}

void ws_transfer_out( struct ws_ccb *item )
{
   int count;
   struct ws_queue *q;

   if ( item->first == NULL )
      return;

   if (( count = write( item->sock, &item->first->buffer->buffer[ item->first->count ], item->first->total )) < 0 )
   {
      syslog( LOG_ERR, "write(): %m" );
      ws_remove_conn( item );
      return;
   }

   item->first->total -= count;
   item->first->count += count;

   if ( ! item->first->total )
   {
      q = item->first;
      item->first = item->first->next;

      if ( ! --q->buffer->refcount )
      {
         free( q->buffer->buffer );
         free( q->buffer );
      }

      free( q );
      --item->qlen;

      if ( item->first == NULL )
      {
         if ( item->closed )
            ws_remove_conn( item );
         else
         {
            item->last = NULL;
            ws_ev_set( item->sock, EVFILT_WRITE, EV_DISABLE, item );
         }
      }
   }
}

int ws_verify_payload_size( struct ws_ccb *item )
{
   /*
    * Client is closing connection.
    */

   if ( item->buffer[ 0 ] == 0x88 )
      return 1;

   /*
    * Frame too long.
    */

   if (( item->len = item->buffer[ 1 ] & 0x7f ) > 126 )
      return 1;

   /*
    * Payload without masking key.
    */

   if ( item->len && ! ( item->buffer[ 1 ] & 0x80 ))
      return 1;

   return 0;
}

int ws_process_payload( struct ws_ccb *item )
{
   int i;
   unsigned char *ptr;

   /*
    * Accept incoming frames from connections we have "closed", but discard
    * the data.  The connection will be closed when all pending outgoing
    * frames including a close frame are delivered to the client.
    */

   if ( item->closed )
      return 0;

   /*
    * If not FIN, discard fragment, close connection.
    */

   if ( !( item->buffer[ 0 ] & 0x80 ))
      return 1;

   /*
    * Unmask payload.
    */

   for( i = 0, ptr = &item->buffer[ item->offset ]; i < item->len; ++ptr, ++i )
      *ptr ^= item->mask[ i % 4 ];

   /*
    * Terminate payload.
    */

   item->buffer[ item->len + item->offset ] = '\0';

   /*
    * Queue pong for ping.
    */

   if ( item->buffer[ 0 ] == 0x89 )
   {
      if ( ws_queue_pong( item ))
         return 1;

      return 0;
   }

   ws_read_callback( item, (( item->buffer[ 0 ] & 0x0f ) == 2 ), item->len, &item->buffer[ item->offset ] );
   return 0;
}

void ws_transfer_in( struct ws_ccb *item )
{
   int i;

   switch( item->stage )
   {
      /*
       * Read first byte of header.  Guaranteed to be ready because
       * we are processing a read event.
       */

      case 0:

         if ( read( item->sock, item->buffer, 1 ) <= 0 )
         {
            ws_remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read second byte of header.
       */

      case 1:

         if (( i = read( item->sock, &item->buffer[ 1 ], 1 )) <= 0 )
         {
            if ( i < 0 && errno == EWOULDBLOCK )
               return;

            ws_remove_conn( item );
            return;
         }

         if ( ws_verify_payload_size( item ))
         {
            ws_remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read first byte of extended length.
       */

      case 2:

         if ( item->len == 126 && ( i = read( item->sock, &item->buffer[ 2 ], 1 )) <= 0 )
         {
            if ( i < 0 && errno == EWOULDBLOCK )
               return;

            ws_remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read second byte of extended length.  Verify payload will fit in buffer with
       * header and terminator.
       */

      case 3:

         if ( item->len == 126 )
         {
            if (( i = read( item->sock, &item->buffer[ 3 ], 1 )) <= 0 )
            {
               if ( i < 0 && errno == EWOULDBLOCK )
                  return;

               ws_remove_conn( item );
               return;
            }

            item->len = ( item->buffer[ 2 ] << 8 ) + item->buffer[ 3 ];

            if ( item->len > ws_buffer_size - 5 )
            {
               ws_remove_conn( item );
               return;
            }
         }

         ++item->stage;
         item->total = 4;
         item->offset = 0;

      /*
       * Read masking key.
       */

      case 4:

         if (( i = read( item->sock, &item->mask[ item->offset ], item->total )) <= 0 )
         {
            if ( i < 0 && errno == EWOULDBLOCK )
               return;

            ws_remove_conn( item );
            return;
         }

         item->offset += i;
         item->total -= i;

         if ( item->total )
            return;

         if ( ! ( item->total = item->len ))
         {
            item->stage = 0;

            /*
             * Ignore empty payload frames: data or pongs. Close frames
             * have been handled earlier.  Return pongs for empty payload
             * pings.
             */

            if ( item->buffer[ 0 ] == 0x89 )
               ws_queue_pong( item );

            return;
         }

         item->offset = ( item->len > 125 ? 4 : 2 );

         if (( item->buffer = realloc( item->buffer, item->offset + item->total + 1 )) == NULL )
         {
            ws_remove_conn( item );
            return;
         }

         ++item->stage;

      /*
       * Read payload.
       */

      case 5:

         if (( i = read( item->sock, &item->buffer[ item->offset ], item->total )) <= 0 )
         {
            if ( i < 0 && errno == EWOULDBLOCK )
               return;

            ws_remove_conn( item );
            return;
         }

         item->offset += i;
         item->total -= i;

         if ( item->total )
            return;

         item->stage = 0;
         item->offset = ( item->len > 125 ? 4 : 2 );

         if ( item->buffer[ 0 ] == 0x8a )
            return;

         if ( ws_process_payload( item ))
            ws_remove_conn( item );
   }
}

void ws_sigterm_handler( int signo )
{
   ++ws_sigterm;
}

void ws_set_sigterm_intr()
{
   struct sigaction sigact;

   sigact.sa_handler = ws_sigterm_handler;
   sigemptyset( &sigact.sa_mask );
   sigact.sa_flags = 0;

   if ( sigaction( SIGTERM, &sigact, NULL ) < 0 )
      syslog( LOG_ERR, "sigaction(): %m" );
}

void ws_process_clients()
{
   int kq;

   if (( kq = kqueue()) < 0 )
   {
      syslog( LOG_ERR, "kqueue(): %m" );
      exit( 1 );
   }

   ws_ev_set( ws_fd, EVFILT_READ, EV_ADD | EV_ENABLE, NULL );

   for( ; ; )
   {
      if ( ws_sigterm )
         break;

      ws_set_sigterm_intr();
      ws_out = kevent( kq, ws_inqueue, ws_in, ws_outqueue, ws_qlen, NULL );
      ws_in = 0;

      if ( ws_out <= 0 )
      {
         if ( errno == EINTR )
            continue;

         syslog( LOG_ERR, "kevent(): %m" );
         exit( 1 );
      }

      signal( SIGTERM, ws_sigterm_handler );

      for( ws_idx = 0; ws_idx < ws_out; ++ws_idx )
      {
         struct ws_ccb *item;

         if ( ! ws_outqueue[ ws_idx ].ident || ws_outqueue[ ws_idx ].flags & EV_ERROR )
            continue;

         item = ( struct ws_ccb *)ws_outqueue[ ws_idx ].udata;

         if ( ws_outqueue[ ws_idx ].filter == EVFILT_TIMER )
         {
            if ( ws_periodic != NULL )
               ws_periodic();
         }
         else if ( ws_outqueue[ ws_idx ].ident == ws_fd )
            ws_accept_connection();
         else if ( ws_outqueue[ ws_idx ].filter == EVFILT_READ )
         {
            if ( item->stage == -1 )
               read_cookie( item );
            else
               ws_transfer_in( item );
         }
         else
            ws_transfer_out( item );
      }
   }
}

void ws_init( int argc, char **argv )
{
   signal( SIGPIPE, SIG_IGN );
   signal( SIGTERM, ws_sigterm_handler );

   ws_set_options( argc, argv );
   ws_qlen = ws_max_conn * MAX_EVENT;

   if (( ws_inqueue = ws_memory( sizeof( struct kevent ) * ws_qlen)) == NULL )
      exit( 1 );

   if (( ws_outqueue = ws_memory( sizeof( struct kevent ) * ws_qlen )) == NULL )
      exit( 1 );
}

int main( int argc, char **argv )
{
   ws_init( argc, argv );
   ws_init_func();

   openlog( ws_app_name, LOG_PID, LOG_DAEMON );
   ws_logging = 1;

   if ( ! ws_testing )
      ws_become_daemon();

   ws_start_listening_unix();
   ws_change_identity();
   ws_process_clients();
   ws_exit_func();

   return 0;
}
